//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/tests/TestAlertBowl.cc 10997 $
// @(#) $Id: TestAlertBowl.cc 10997 2020-08-27 05:34:16Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file TestAlertBowl.cc
//!  \author Daniel W. McRobb
//!  \brief unit tests for Dwm::Mcrover::AlertBowl
//---------------------------------------------------------------------------

extern "C" {
  #include <fcntl.h>
  #include <unistd.h>
}

#include <iostream>
#include <sstream>

#include "DwmUnitAssert.hh"
#include "DwmDnsMessageQuestion.hh"
#include "DwmMcroverAlertBowl.hh"
#include "DwmMcroverUnitTestUtils.hh"

using namespace std;
using namespace Dwm;

#define MY_IOTESTFILE "./TestAlertBowl.tmp"

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestAccessors()
{
  using Mcrover::AlertOrigin, Mcrover::AlertBowl, Mcrover::WebAlert;

  AlertOrigin  origin("dwm", Dwm::Ipv4Address("192.168.0.1"));
  WebAlert     alert("http://somehost/index.html", 404);
  AlertBowl    bowl;
  UnitAssert(bowl.Add(origin, alert));
  UnitAssert(bowl.Remove(origin, alert));
  UnitAssert(! bowl.Remove(origin, alert));
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestBasicIO()
{
  using Mcrover::AlertOrigin, Mcrover::AlertBowl, Mcrover::AlertBowlValue,
    Mcrover::WebAlert, Mcrover::UPSAlert, Mcrover::UPSStatus,
    Mcrover::DNSAlert;
  
  AlertBowl        bowl;
  AlertOrigin      origin("somepack", Dwm::Ipv4Address("192.168.168.3"));
  WebAlert  webAlert("http://zyx.onm.abc/index.html", 404);
  UnitAssert(bowl.Add(origin, webAlert));
  UPSAlert  upsAlert("myups", Ipv4Address("10.2.3.4"), "upsName",
                     UPSStatus(UPSStatus::e_upsOnBattery));
  UnitAssert(bowl.Add(origin, upsAlert));
  DNSAlert  dnsAlert(Ipv4Address("9.9.9.9"), "www.google.com",
                     Dwm::Dns::MessageQuestion::k_typeA);
  UnitAssert(bowl.Add(origin, dnsAlert));
  origin.PackName("rover1");
  origin.MemberAddress(Dwm::Ipv4Address("192.168.168.2"));
  UnitAssert(bowl.Add(origin, webAlert));
  UnitAssert(bowl.Add(origin, upsAlert));
  vector<pair<time_t,string>>  conciseAlerts;
  UnitAssert(bowl.GetConcise(conciseAlerts));

  AlertBowl  packBowl1;
  UnitAssert(! bowl.GetByPack("noexistpack", packBowl1));
  UnitAssert(bowl.GetByPack("somepack", packBowl1));
  UnitAssert(packBowl1.GetConcise(conciseAlerts));
  UnitAssert(conciseAlerts.size() == 3);

  AlertBowl  originBowl1;
  UnitAssert(! bowl.GetByOrigin(AlertOrigin("nono",Ipv4Address("5.6.7.8")),
                                originBowl1));
  UnitAssert(bowl.GetByOrigin(AlertOrigin("somepack",Ipv4Address("192.168.168.3")),
                              originBowl1));
  UnitAssert(originBowl1.GetConcise(conciseAlerts));
  UnitAssert(conciseAlerts.size() == 3);
  
  Mcrover::UnitTestUtils::TestBasicDescriptorIO(bowl, MY_IOTESTFILE);
  Mcrover::UnitTestUtils::TestBasicStreamIO(bowl);
  Mcrover::UnitTestUtils::TestBasicFileIO(bowl, MY_IOTESTFILE);
  return;
}
      
//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  TestAccessors();
  TestBasicIO();
  
  if (Assertions::Total().Failed()) {
    Assertions::Print(cerr, true);
  }
  else {
    cout << Assertions::Total() << " passed" << endl;
  }
  
}
