//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/src/DwmMcroverRPCAlert.cc 11886 $
// @(#) $Id: DwmMcroverRPCAlert.cc 11886 2021-04-07 05:16:39Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverRPCAlert.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::RPCAlert class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <netdb.h>
}

#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmIOUtils.hh"
#include "DwmStreamIO.hh"
#include "DwmSvnTag.hh"
#include "DwmMcroverRPCAlert.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/src/DwmMcroverRPCAlert.cc 11886 $");

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RPCAlert::RPCAlert(const std::string	& name,
                       const Ipv4Address & addr,
                       const RPCProgramId & program)
        : _data({addr, program, name})
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & RPCAlert::Name() const
    {
      return std::get<2>(_data);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string &	RPCAlert::Name(const std::string & name)
    {
      std::get<2>(_data) = name;
      return std::get<2>(_data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & RPCAlert::Address() const
    {
      return std::get<0>(_data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & RPCAlert::Address(const Ipv4Address & addr)
    {
      std::get<0>(_data) = addr;
      return std::get<0>(_data);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const RPCProgramId & RPCAlert::Program() const
    {
      return std::get<1>(_data);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const RPCProgramId & RPCAlert::Program(const RPCProgramId & program)
    {
      std::get<1>(_data) = program;
      return std::get<1>(_data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json RPCAlert::ToJson() const
    {
      nlohmann::json  rc;
      rc["name"] = std::get<2>(_data);
      rc["addr"] = (std::string)std::get<0>(_data);
      rc["prog"] = std::get<1>(_data).ToJson();
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCAlert::FromJson(const nlohmann::json & j)
    {
      bool  rc = false;
      if (j.is_object()) {
        auto  name = j.find("name");
        if ((name != j.end()) && name->is_string()) {
          Name(name->get<std::string>());
          auto  addr = j.find("addr");
          if ((addr != j.end()) && addr->is_string()) {
            Address(Ipv4Address(addr->get<std::string>()));
            auto  prog = j.find("prog");
            if ((prog != j.end()) && prog->is_object()) {
              rc = std::get<1>(_data).FromJson(*prog);
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string RPCAlert::DisplayString() const
    {
      std::string  rc;
      if (! Name().empty()) {
        rc += Name() + ':';
      }
      else {
        rc += (std::string)Address() + ':';
      }
      rc += Program().Name() + ":v";
      rc += std::to_string(Program().Version()) + ':';
      struct protoent *protoEntry = getprotobynumber(Program().Protocol());
      if (protoEntry) {
        rc += protoEntry->p_name;
      }
      else {
        rc += std::to_string(Program().Protocol());
      }
      rc += " RPC";
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & RPCAlert::Read(std::istream & is)
    {
      Address(INADDR_NONE);
      Name("");
      return Dwm::StreamIO::Read(is, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t RPCAlert::Read(int fd)
    {
      Address(INADDR_NONE);
      Name("");
      return Dwm::DescriptorIO::Read(fd, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t RPCAlert::Read(FILE * f)
    {
      Address(INADDR_NONE);
      Name("");
      return Dwm::FileIO::Read(f, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & RPCAlert::Write(std::ostream & os) const
    {
      return Dwm::StreamIO::Write(os, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t RPCAlert::Write(int fd) const
    {
      return Dwm::DescriptorIO::Write(fd, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t RPCAlert::Write(FILE *f) const
    {
      return Dwm::FileIO::Write(f, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t RPCAlert::StreamedLength() const
    {
      return Dwm::IOUtils::StreamedLength(_data);
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm
