%code requires
{
  #include <string>
  #include <vector>
  #include <boost/asio.hpp>

  #include "DwmIpPrefix.hh"
  #include "DwmMcroverRangeSetPair.hh"

  using std::vector, std::set, std::string, std::pair,
      Dwm::Ipv4Address, Dwm::Ipv4Prefix, Dwm::Ipv6Address;

  extern "C" {
    extern int mcrovercfglex_destroy(void);
  }
}

%{
  #include <cstdio>

  extern "C" {
    #include <netdb.h>

    extern void mcrovercfgerror(const char *arg, ...);
    extern FILE *mcrovercfgin;
  }
        
  #include <string>
  #include <vector>

  #include "DwmStreamIO.hh"
  #include "DwmSvnTag.hh"
  #include "DwmSysLogger.hh"
  #include "DwmDnsUtils.hh"
  #include "DwmMcroverConfig.hh"
  #include "DwmMcroverUtils.hh"

  using namespace std;
  
  string                g_configPath;
  Dwm::Mcrover::Config  *g_config = nullptr;

  static const vector<pair<string,uint16_t>> supportedQTypes = {
    { "A",     Dwm::Dns::MessageQuestion::k_typeA },
    { "NS",    Dwm::Dns::MessageQuestion::k_typeNS },
    { "CNAME", Dwm::Dns::MessageQuestion::k_typeCNAME },
    { "SOA",   Dwm::Dns::MessageQuestion::k_typeSOA },
    { "PTR",   Dwm::Dns::MessageQuestion::k_typePTR },
    { "MX",    Dwm::Dns::MessageQuestion::k_typeMX },
    { "AAAA",  Dwm::Dns::MessageQuestion::k_typeAAAA },
    { "LOC",   Dwm::Dns::MessageQuestion::k_typeLOC }
  };
  
%}

%define api.prefix {mcrovercfg}

%union {
  bool                                           boolVal;
  int                                            intVal;
  uint32_t                                       uint32Val;
  uint64_t                                       uint64Val;
  float                                          floatVal;
  string                                        *stringVal;
  vector<string>                                *stringVecVal;
  vector<uint16_t>                              *uint16VecVal;
  Ipv4Address                                   *ipv4AddrVal;
  Ipv6Address                                   *ipv6AddrVal;
  Dwm::Mcrover::RPCProgramId                    *rpcProgIdVal;
  vector<Dwm::Mcrover::RPCProgramId>            *rpcProgsVecVal;
  Dwm::Mcrover::TargetHostConfig                *targetHostConfigVal;
  vector<Dwm::Mcrover::TargetHostConfig>        *targetHostConfigVecVal;
  Dwm::Dns::MessageQuestion                     *dnsQuestionVal;
  vector<Dwm::Dns::MessageQuestion>             *dnsQuestionVecVal;
  pair<string,uint8_t>                          *stringUint8PairVal;
  vector<pair<string,uint8_t>>                  *stringUint8PairVecVal;
  vector<Dwm::Ipv4Prefix>                       *ipv4PrefixVecVal;
  pair<Ipv4Prefix,vector<Dwm::Ipv4Prefix>>      *ipv4PrefixVecPairVal;
  vector<pair<Ipv4Prefix,vector<Ipv4Prefix>>>   *ipv4PrefixVecVecVal;
  vector<Dwm::Mcrover::PackConfig>              *packConfigVecVal;
  Dwm::Mcrover::PackConfig                      *packConfigVal;
  Dwm::Mcrover::PackMemberConfig                *packMemberConfigVal;
  vector<Dwm::Mcrover::PackMemberConfig>        *packMemberConfigVecVal;
  pair<uint16_t,uint16_t>                       *uint16PairVal;
  Dwm::Mcrover::RangeSetPair<uint16_t>          *rangeSetPairUint16Val;
  Dwm::Mcrover::WebTarget                       *webTargetVal;
  vector<Dwm::Mcrover::WebTarget>               *webTargetVecVal;
  pair<float,float>                             *floatPairVal;
  Dwm::Mcrover::WeatherConfig                   *weatherConfigVal;
  set<Dwm::IpPrefix>                            *ipPrefixSetVal;
  Dwm::Mcrover::ServiceConfig                   *serviceConfigVal;
  boost::asio::ip::tcp::endpoint                *serviceAddrVal;
  std::set<boost::asio::ip::tcp::endpoint>      *serviceAddrSetVal;
  pair<string,string>                           *stringStringPairVal;
  std::map<string,string>                       *stringStringMapVal;
  Dwm::Mcrover::WebAppTarget                    *webAppTargetVal;
  vector<Dwm::Mcrover::WebAppTarget>            *webAppTargetVecVal;
  Dwm::Mcrover::TcpDstAddr                      *tcpDstAddrVal;
  vector<Dwm::Mcrover::TcpDstAddr>              *tcpDstAddrVecVal;
  pair<string,uint64_t>                         *stringUint64PairVal;
  vector<pair<string,uint64_t>>                 *stringUint64PairVecVal;
}

%code provides
{
  // Tell Flex the expected prototype of yylex.
  #define YY_DECL                             \
    int mcrovercfglex ()

  // Declare the scanner.
  YY_DECL;
}

%token ADDRESS ADDRESSES ALERTS ALLOWEDCLIENTS APPNAME BANDITS CAPACITY
%token CREDENCE DEVICE DISKS DNS DUMPS FALSESTRING FS GUESTS FACILITY HOST
%token IPV4 IPV6 KEYDIRECTORY LATITUDE LEVEL LOCAL LOCATION LOGLOCATIONS
%token LONGITUDE MOUNT NAME MEMBERS OTHERPACKS PACK PARAMS PERIOD PORT
%token PROTOCOL QNAME QTYPE RESOLVE ROUTES RPC RPC6 SERVERS SERVICE SMTP
%token SMTP6 STATIONS STATUS STORE STRIPDOMAINS SYSLOG TCP4 TCP6 TCP4DENIED
%token TCP6DENIED TRUESTRING UPS URI VALIDATECERT VERSION WEATHER WEB
%token WEBAPP XPATH ZFS ZPOOL

%token<stringVal>  STRING
%token<intVal>     INTEGER
%token<uint64Val>  TIMEPERIOD

%type<boolVal>                 ValidateCertificate
%type<ipv4AddrVal>             Ipv4AddrStanza
%type<ipv6AddrVal>             Ipv6AddrStanza
%type<intVal>                  TCP4Port ZpoolCapacity
%type<intVal>                  FileSystemCapacity
%type<uint32Val>               RPCProgramVersion RPCProgramProtocol
%type<rpcProgsVecVal>          RPCStanza RPC6Stanza RPCPrograms
%type<rpcProgIdVal>            RPCProgram RPCProgramId
%type<stringVal>               AlertFile AppName Xpath RPCProgramName
%type<stringVal>               KeyDirectory
%type<stringVal>               DNSQuestionQName DNSQuestionQType ZpoolName
%type<stringVal>               PackName MemberName
%type<stringVal>               FileSystemMount Uri DiskDevice DiskName
%type<stringVecVal>            UPSStanza RouteGateways
%type<stringVecVal>            VectorOfString AppParams
%type<targetHostConfigVecVal>  Servers ServerList
%type<targetHostConfigVal>     Server ServerData
%type<uint16VecVal>            TCP4PortStanza TCP6PortStanza
%type<uint16VecVal>            TCP4PortDeniedStanza TCP6PortDeniedStanza
%type<uint16VecVal>            TCP4Ports
%type<uint16VecVal>            SMTPStanza SMTP6Stanza
%type<dnsQuestionVal>          DNSQuestion DNSQuestionData
%type<dnsQuestionVecVal>       DNSQuestions DNSResolve DNSStanza
%type<stringUint8PairVal>      ZpoolSetting Zpool FileSystemSetting FileSystem
%type<stringUint8PairVecVal>   Zpools FileSystemList
%type<ipv4PrefixVecPairVal>    Route
%type<ipv4PrefixVecVecVal>     Routes
%type<packMemberConfigVal>     Member MemberInfo
%type<packMemberConfigVecVal>  MembersList Members Pack
%type<packConfigVecVal>        OtherPackEntries
%type<packConfigVal>           PackContents OtherPackEntry
%type<rangeSetPairUint16Val>   WebResponses IntegerRanges IntegerRangeSet
%type<uint16PairVal>           IntegerRangePair
%type<webTargetVal>            WebTarget WebTargetParts
%type<webTargetVecVal>         WebStanza WebTargets
%type<weatherConfigVal>        WeatherHost
%type<serviceConfigVal>        ServiceSettings
%type<serviceAddrSetVal>       ServiceAddresses ServiceAddressSet
%type<serviceAddrSetVal>       MemberAddresses
%type<ipPrefixSetVal>          AllowedClients
%type<serviceAddrVal>          ServiceAddress
%type<stringStringPairVal>     Disk
%type<stringStringMapVal>      DiskList
%type<webAppTargetVal>         WebAppTarget WebAppTargetParts
%type<webAppTargetVecVal>      WebAppStanza WebAppTargets
%type<tcpDstAddrVal>           CredenceServiceTarget CredenceServiceTargetParts
%type<tcpDstAddrVecVal>        CredenceStanza CredenceServiceTargets
%type<stringUint64PairVal>     DumpDevice
%type<stringUint64PairVecVal>  DumpDeviceList
%type<uint64Val>               DumpPeriod

%%

Config: TopStanza | Config TopStanza;

TopStanza: Alerts | Service | Syslog | Local | Servers | Pack | OtherPacks | Weather;

Alerts: ALERTS '{' AlertFile '}' ';'
{
  if (g_config) {
     g_config->AlertFile(*$3);
  }
  delete $3;
};

AlertFile: STORE '=' STRING ';'
{
  $$ = $3;
};

Syslog: SYSLOG '{' SyslogSettings '}' ';';

SyslogSettings: SyslogSetting | SyslogSettings SyslogSetting;

SyslogSetting: SyslogFacility | SyslogLevel | SyslogLocations
{};

SyslogLevel: LEVEL '=' STRING ';'
{
  if (g_config) {
    g_config->SyslogLevel(*$3);
  }
  delete $3;
};

SyslogFacility: FACILITY '=' STRING ';'
{
  if (g_config) {
    g_config->SyslogFacility(*$3);
  }
  delete $3;
};

SyslogLocations: LOGLOCATIONS '=' STRING ';'
{
  if (g_config) {
    if (strncasecmp($3->c_str(), "yes", 3) == 0) {
      g_config->SyslogLocations(true);
    }
  }
  delete $3;
};

Service: SERVICE '{' ServiceSettings '}' ';'
{
  if (g_config) {
    g_config->RunService(true);
    g_config->Service(*$3);
  }
  delete $3;
};

ServiceSettings: ServiceAddresses
{
  $$ = new Dwm::Mcrover::ServiceConfig();
  $$->Addresses(*$1);
  delete $1;
}
| KeyDirectory
{
  $$ = new Dwm::Mcrover::ServiceConfig();
  $$->KeyDirectory(*$1);
  delete $1;
}
| AllowedClients
{
  $$ = new Dwm::Mcrover::ServiceConfig();
  $$->AllowedClients() = *$1;
  delete $1;
}
| ServiceSettings ServiceAddresses
{
  $$->Addresses(*$2);
  delete $2;
}
| ServiceSettings KeyDirectory
{
  $$->KeyDirectory(*$2);
  delete $2;
}
| ServiceSettings AllowedClients
{
  $$->AllowedClients() = *$2;
  delete $2;
};

ServiceAddresses: ADDRESSES '=' '[' ServiceAddressSet ']' ';'
{
    $$ = $4;
};

ServiceAddressSet: ServiceAddress
{
  $$ = new std::set<boost::asio::ip::tcp::endpoint>();
  $$->insert(*$1);
  delete $1;
}
| ServiceAddressSet ',' ServiceAddress
{
  $$->insert(*$3);
  delete $3;
};

ServiceAddress: '{' ADDRESS '=' STRING ';' '}'
{
  using batcp = boost::asio::ip::tcp;
  if (*$4 == "in6addr_any") {
    $$ = new batcp::endpoint(batcp::v6(), 2123);
  }
  else if (*$4 == "inaddr_any") {
    $$ = new batcp::endpoint(batcp::v4(), 2123);
  }
  else {
    boost::system::error_code  ec;
    boost::asio::ip::address  addr =
    boost::asio::ip::address::from_string(*$4, ec);
    if (ec) {
      mcrovercfgerror("invalid IP address");
      delete $4;
      return 1;
    }
    $$ = new boost::asio::ip::tcp::endpoint(addr, 2123);
  }
  delete $4;
}
| '{' ADDRESS '=' STRING ';' PORT '=' TCP4Port ';' '}'
{
  namespace baip = boost::asio::ip;
  using batcp =	boost::asio::ip::tcp;
  
  if (($8 <= 0) || ($8 > 65535)) {
    mcrovercfgerror("invalid port");
    delete $4;
    return 1;
  }

  if (*$4 == "in6addr_any") {
      $$ = new batcp::endpoint(batcp::v6(), $8);
  }
  else if (*$4 == "inaddr_any") {
      $$ = new batcp::endpoint(batcp::v4(), $8);
  }
  else {
    boost::system::error_code  ec;
    baip::address  addr = baip::address::from_string(*$4, ec);
    if (ec) {
      mcrovercfgerror("invalid IP address");
      delete $4;
      return 1;
    }
    $$ = new batcp::endpoint(addr, $8);
  }
  delete $4;
}
| '{' PORT '=' TCP4Port ';' ADDRESS '=' STRING ';' '}'
{
  namespace baip = boost::asio::ip;
  using batcp = boost::asio::ip::tcp;

  if (($4 <= 0) || ($4 > 65535)) {
    mcrovercfgerror("invalid port");
    delete $8;
    return 1;
  }
  baip::address  addr;
  if (*$8 == "in6addr_any") {
    $$ = new batcp::endpoint(batcp::v6(), $4);
  }
  else if (*$8 == "inaddr_any") {
    $$ = new batcp::endpoint(batcp::v4(), $4);
  }
  else {
    boost::system::error_code  ec;
    baip::address addr = baip::address::from_string(*$8, ec);
    if (ec) {
      mcrovercfgerror("invalid IP address");
      delete $8;
      return 1;
    }
    $$ = new batcp::endpoint(addr, $4);
  }
  delete $8;
};

KeyDirectory : KEYDIRECTORY '=' STRING ';'
{
  $$ = $3;
};

AllowedClients: ALLOWEDCLIENTS '=' '[' VectorOfString ']' ';'
{
  $$ = new std::set<Dwm::IpPrefix>();
  for (const auto & pfxstr : *$4) {
    Dwm::IpPrefix   pfx(pfxstr);
    if (pfx.Family() != AF_INET) {
      $$->insert(pfx);
    }
    else {
      if (pfx.Prefix<Ipv4Prefix>()->Network().Raw() != INADDR_NONE) {
        $$->insert(pfx);
      }
      else {
        mcrovercfgerror("invalid IP prefix");
        delete $4;
        return 1;
      }
    }
  }
  delete $4;
}
|
{};

OtherPacks: OTHERPACKS '=' '[' OtherPackEntries ']' ';'
{
  for (const auto & pe : *($4)) {
    g_config->OtherPacks()[pe.PackName()] = pe;
  }
  delete $4;
};

OtherPackEntries: OtherPackEntry
{
  $$ = new std::vector<Dwm::Mcrover::PackConfig>();
  $$->push_back(*$1);
  delete $1;
}
| OtherPackEntries ',' OtherPackEntry
{
  $$->push_back(*$3);
  delete $3;
};

OtherPackEntry: '{' PackContents '}'
{
  $$ = $2;
};
    
Pack: PACK '{' PackContents '}' ';'
{
  if (g_config) {
      g_config->MyPack(*$3);
  }
  delete $3;
};

PackContents: PackName
{
  $$ = new Dwm::Mcrover::PackConfig();
  $$->PackName(*$1);
  delete $1;
}
| MembersList
{
  $$ = new Dwm::Mcrover::PackConfig();
  for (const auto & member : *$1) {
    $$->Members()[member.Name()] = member;
  }
  delete $1;
}
| PackContents PackName
{
  $$->PackName(*$2);
  delete $2;
}
| PackContents MembersList
{
  for (const auto & member : *$2) {
    $$->Members()[member.Name()] = member;
  }
  delete $2;
};

PackName: NAME '=' STRING ';'
{
  $$ = $3;
};

MembersList: MEMBERS '=' '[' Members ']' ';'
{
  $$ = $4;
};

Members: Member
{
  $$ = new vector<Dwm::Mcrover::PackMemberConfig>();
  $$->push_back(*$1);
  delete $1;
}
| Members ',' Member
{
  $$->push_back(*$3);
  delete $3;
};

Member: '{' MemberInfo '}'
{
  $$ = $2;
};

MemberInfo : MemberName
{
  $$ = new Dwm::Mcrover::PackMemberConfig();
  $$->Name(*$1);
  delete $1;
}
| MemberAddresses
{
  $$ = new Dwm::Mcrover::PackMemberConfig();
  $$->Addresses(*$1);
  delete $1;
}
| MemberInfo MemberName
{
  $$->Name(*$2);
  delete $2;
}
| MemberInfo MemberAddresses
{
  $$->Addresses(*$2);
  delete $2;
};

MemberAddresses: ADDRESSES '=' '[' ServiceAddressSet ']' ';'
{
  $$ = $4;
};

MemberName: NAME '=' STRING ';'
{
  $$ = $3;
};

Local: LOCAL '{' LocalSettings '}' ';'
{};

LocalSettings: LocalSetting
{}
| LocalSettings LocalSetting
{};

LocalSetting : ZpoolStanza | ZFSStanza | FileSystemStanza | DisksStanza |
               | DumpsStanza | RoutesStanza | GuestStanza | BanditStanza
{};

GuestStanza: GUESTS '=' '[' VectorOfString ']' ';'
{
  if (g_config) {
    for (const auto & pfxstr : *$4) {
      Ipv4Prefix   pfx(pfxstr);
      if (pfx.Network().Raw() != INADDR_NONE) {
        g_config->Local().Guests().push_back(pfx);
      }
      else {
        mcrovercfgerror("invalid IPv4 prefix");
        delete $4;
        return 1;
      }
    }
  }
  delete $4;
};

BanditStanza: BANDITS '=' '[' VectorOfString ']' ';'
{
  if (g_config) {
    for (const auto & pfxstr : *$4) {
      Ipv4Prefix   pfx(pfxstr);
      if (pfx.Network().Raw() != INADDR_NONE) {
        g_config->Local().Bandits().push_back(pfx);
      }
      else {
        mcrovercfgerror("invalid IPv4 prefix");
        delete $4;
        return 1;
      }
    }
  }
  delete $4;
};

RoutesStanza: ROUTES '=' '[' Routes ']' ';'
{
  if (g_config) {
    for (const auto & rt : *$4) {
      g_config->Local().Routes()[rt.first] = rt.second;
    }
  }
  delete $4;
};

Routes: Route
{
  $$ =
    new vector<pair<Ipv4Prefix,vector<Ipv4Prefix>>>();
  $$->push_back(*$1);
  delete $1;
}
| Routes ',' Route
{
  $$->push_back(*$3);
  delete $3;
};

Route: '{' STRING '=' RouteGateways ';' '}'
{
  $$ = new pair<Ipv4Prefix,vector<Ipv4Prefix>>();
  if (*$2 != "default") {
    $$->first = Dwm::Ipv4Prefix(*$2);
  }
  else {
    $$->first = Dwm::Ipv4Prefix("0/0");
  }
  for (const auto & gw : *$4) {
    $$->second.push_back(Ipv4Prefix(gw));
  }
  delete $2;
  delete $4;  
};

RouteGateways: '[' VectorOfString ']'
{
  $$ = $2;
};

ZpoolStanza: ZPOOL '=' '[' Zpools ']' ';'
{
  if (g_config) {
    for (const auto & pool : *$4) {
      g_config->Local().Zpools()[pool.first] = pool.second;
    }
  }
  delete $4;
};

Zpools: Zpool
{
  $$ = new vector<pair<string,uint8_t>>();
  $$->push_back(*$1);
  delete $1;
}
| Zpools ',' Zpool {
  $$->push_back(*$3);
  delete $3;
};

Zpool : '{' ZpoolSetting '}'
{
  $$ = $2;
};

ZpoolSetting : ZpoolName
{
  $$ = new pair<string,uint8_t>(*$1, 255);
  delete $1;
}
| ZpoolCapacity
{
  $$ = new pair<string,uint8_t>("", $1);
}
| ZpoolSetting ZpoolName
{
  $$->first = *$2;
  delete $2;
}
| ZpoolSetting ZpoolCapacity
{
  $$->second = $2;
};

ZpoolCapacity : CAPACITY '=' INTEGER ';'
{
  if (($3 >= 0) && ($3 <= 100)) {
    $$ = $3;
  }
  else {
    mcrovercfgerror("invalid zpool capacity");
    return 1;
  }
};

ZpoolName : NAME '=' STRING ';'
{
  $$ = $3;
};

ZFSStanza: ZFS '=' '[' FileSystemList ']' ';'
{
    delete $4;
};

FileSystemStanza : FS '=' '[' FileSystemList ']' ';'
{
  if (g_config) {
    for (const auto & fs : *$4) {
      g_config->Local().Filesystems()[fs.first] = fs.second;
    }
  }
  delete $4;
};

FileSystemList : FileSystem
{
  $$ = new vector<pair<string,uint8_t>>();
  $$->push_back(*$1);
  delete $1;
}
| FileSystemList ',' FileSystem
{
  $$->push_back(*$3);
  delete $3;
};

FileSystem : '{' FileSystemSetting '}'
{
  $$ = $2;
};

FileSystemSetting : FileSystemMount
{
  $$ = new pair<string,uint8_t>(*$1,255);
  delete $1;
}
| FileSystemCapacity
{
  $$ = new pair<string,uint8_t>("", $1);  
}
| FileSystemSetting FileSystemMount
{
  $$->first = *$2;
  delete $2;
}
| FileSystemSetting FileSystemCapacity
{
  $$->second = $2;
};

FileSystemMount : MOUNT '=' STRING ';'
{
  $$ = $3;
};

FileSystemCapacity: CAPACITY '=' INTEGER ';'
{
  if (($3 >= 0) && ($3 <= 100)) {
    $$ = $3;
  }
  else {
    mcrovercfgerror("invalid filesystem capacity");
    return 1;
  }
};

DumpsStanza : DUMPS '=' '[' DumpDeviceList ']' ';'
{
  if (g_config) {
    for (const auto & fs : *$4) {
      g_config->Local().Dumps()[fs.first] = fs.second;
    }
  }
  delete $4;
};

DumpDeviceList : DumpDevice
{
  $$ = new vector<pair<string,uint64_t>>();
  $$->push_back(*$1);
  delete $1;
}
| DumpDeviceList ',' DumpDevice
{
  $$->push_back(*$3);
  delete $3;
};

DumpDevice : '{' DiskDevice DumpPeriod '}'
{
  $$ = new pair<string,uint64_t>(*$2,$3);
  delete $2;
}
| '{' DumpPeriod DiskDevice '}'
{
  $$ = new pair<string,uint64_t>(*$3,$2);
  delete $3;
};

DumpPeriod: PERIOD '=' TIMEPERIOD ';'
{
    $$ = $3;
};

DisksStanza: DISKS '=' '[' DiskList ']' ';'
{
  if (g_config) {
    for (const auto & disk : *$4) {
      g_config->Local().Disks()[disk.first] = disk.second;
    }
  }
  delete $4;
};

DiskList: Disk
{
  $$ = new map<string,string>();
  (*$$)[$1->first] = $1->second;
  delete $1;
}
| DiskList ',' Disk
{
  (*$$)[$3->first] = $3->second;
  delete $3;
};

Disk: '{' DiskName DiskDevice '}'
{
  $$ = new pair<string,string>(*$2,*$3);
  delete $2;
  delete $3;
}
| '{' DiskDevice DiskName '}'
{
  $$ = new pair<string,string>(*$3,*$2);
  delete $2;
  delete $3;
};

DiskName: NAME '=' STRING ';'
{
  $$ = $3;
};

DiskDevice: DEVICE '=' STRING ';'
{
  $$ = $3;
};

Servers: SERVERS '{' ServerList '}' ';'
{
  if (g_config) {
    g_config->Servers(*$3);
  }
  delete $3;
};

ServerList: Server
{
  $$ = new vector<Dwm::Mcrover::TargetHostConfig>();
  $$->push_back(*$1);
  delete $1;
}
| ServerList Server
{
  $$->push_back(*$2);
  delete $2;
};

Server: STRING '{' ServerData '}' ';'
{
  $$ = $3;
  $$->Name(*$1);
  vector<Dwm::Mcrover::TcpDstAddr>  credenceTargets;
  for (auto it = $$->CredencePeerTargets().begin();
       it != $$->CredencePeerTargets().end(); ++it) {
      if (it->Address() == Dwm::IpAddress(Dwm::Ipv4Address(INADDR_NONE))) {
          //  Credence entry doesn't specify address.  Use host addresses.
          if ($$->Address() != Dwm::Ipv4Address(INADDR_NONE)) {
              Dwm::IpAddress  addr($$->Address());
              Dwm::Mcrover::TcpDstAddr  dstAddr(it->Name(), addr, it->Port());
              credenceTargets.push_back(dstAddr);
          }
          if ($$->Address6() != Dwm::Ipv6Address()) {
              Dwm::IpAddress  addr($$->Address6());
              Dwm::Mcrover::TcpDstAddr	dstAddr(it->Name(), addr, it->Port());
              credenceTargets.push_back(dstAddr);
          }
      }
      else {
          Dwm::Mcrover::TcpDstAddr  dstAddr(it->Name(), it->Address(),
                                            it->Port());
          credenceTargets.push_back(dstAddr);
      }
  }
  $$->CredencePeerTargets(credenceTargets);
              
  delete $1;
};

ServerData: Ipv4AddrStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->Address(*$1);
  delete $1;
}
| TCP4PortStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->TCP4Ports(*$1);
  delete $1;
}
| TCP6PortStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->TCP6Ports(*$1);
  delete $1;
}
| TCP4PortDeniedStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->TCP4PortsDenied(*$1);
  delete $1;
}
| TCP6PortDeniedStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->TCP6PortsDenied(*$1);
  delete $1;
}
| RPCStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->ExpectedPrograms(*$1);
  delete $1;
}
| RPC6Stanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->Expected6Programs(*$1);
  delete $1;
}
| SMTPStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->SMTPPorts(*$1);
  delete $1;
}
| SMTP6Stanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->SMTP6Ports(*$1);
  delete $1;
}
| UPSStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->UPS(*$1);
  delete $1;
}
| WebStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->WebTargets(*$1);
  delete $1;
}
| DNSStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->DNSResolve(*$1);
  delete $1;
}
| WebAppStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->WebAppTargets(*$1);
  delete $1;
}
| CredenceStanza
{
  $$ = new Dwm::Mcrover::TargetHostConfig();
  $$->CredencePeerTargets(*$1);
  delete $1;
}
| ServerData Ipv4AddrStanza        { $$->Address(*$2);  delete $2; }
| ServerData Ipv6AddrStanza        { $$->Address6(*$2);  delete $2; }
| ServerData TCP4PortStanza        { $$->TCP4Ports(*$2); delete $2; }
| ServerData TCP6PortStanza        { $$->TCP6Ports(*$2); delete $2; }
| ServerData TCP4PortDeniedStanza  { $$->TCP4PortsDenied(*$2); delete $2; }
| ServerData TCP6PortDeniedStanza  { $$->TCP6PortsDenied(*$2); delete $2; }
| ServerData RPCStanza             { $$->ExpectedPrograms(*$2); delete $2; }
| ServerData RPC6Stanza            { $$->Expected6Programs(*$2); delete $2; }
| ServerData SMTPStanza            { $$->SMTPPorts(*$2); delete $2; }
| ServerData SMTP6Stanza           { $$->SMTP6Ports(*$2); delete $2; }
| ServerData UPSStanza             { $$->UPS(*$2); delete $2; }
| ServerData WebStanza             { $$->WebTargets(*$2); delete $2; }
| ServerData DNSStanza             { $$->DNSResolve(*$2); delete $2; }
| ServerData WebAppStanza          { $$->WebAppTargets(*$2); delete $2; }
| ServerData CredenceStanza        { $$->CredencePeerTargets(*$2); delete $2; }
;

Ipv6AddrStanza: IPV6 '=' STRING ';'
{
  $$ = new Dwm::Ipv6Address(*$3);
  delete $3;
};

Ipv4AddrStanza: IPV4 '=' STRING ';'
{
  $$ = new Dwm::Ipv4Address(*$3);
  delete $3;
};

TCP6PortStanza: TCP6 '=' '[' TCP4Ports ']' ';'
{
  $$ = $4;
};

TCP4PortStanza: TCP4 '=' '[' TCP4Ports ']' ';'
{
  $$ = $4;
};

TCP4PortDeniedStanza: TCP4DENIED '=' '[' TCP4Ports ']' ';'
{
    $$ = $4;
};

TCP6PortDeniedStanza: TCP6DENIED '=' '[' TCP4Ports ']' ';'
{
    $$ = $4;
};

TCP4Ports: TCP4Port
{
  $$ = new vector<uint16_t>();
  $$->push_back($1);
}
| TCP4Ports ',' TCP4Port    { $$->push_back($3); }
;

TCP4Port: INTEGER
{
  if (($1 > 0) && ($1 < 65536)) {
    $$ = $1;
  }
  else {
    mcrovercfgerror("invalid TCP port number");
    return 1;
  }
}
| STRING
{
  auto  servEntry = getservbyname($1->c_str(), "tcp");
  if (servEntry) {
    $$ = ntohs(servEntry->s_port);
  }
  else {
      mcrovercfgerror("unknown TCP service");
      delete $1;
      return 1;
  }
  delete $1;
};

RPCStanza: RPC '=' '[' RPCPrograms ']' ';'    { $$ = $4; };

RPC6Stanza: RPC6 '=' '[' RPCPrograms ']' ';'    { $$ = $4; };

RPCPrograms: RPCProgram
{
  $$ = new vector<Dwm::Mcrover::RPCProgramId>();
  $$->push_back(*$1);
  delete $1;
}
| RPCPrograms ',' RPCProgram
{
  $$->push_back(*$3);
  delete $3;
}
;

RPCProgram: '{' RPCProgramId '}'
{
  $$ = $2;
  *($$) = Dwm::Mcrover::RPCProgramId($$->Name(), $$->Protocol(), $$->Version());
};

RPCProgramId: RPCProgramName
{
  $$ = new Dwm::Mcrover::RPCProgramId();
  $$->Name(*$1);
  delete $1;
}
| RPCProgramProtocol
{
  $$ = new Dwm::Mcrover::RPCProgramId(); $$->Protocol($1);
}
| RPCProgramVersion { $$ = new Dwm::Mcrover::RPCProgramId(); $$->Version($1); }
| RPCProgramId  RPCProgramName      { $$->Name(*$2); delete $2; }
| RPCProgramId  RPCProgramProtocol  { $$->Protocol($2); }
| RPCProgramId  RPCProgramVersion   { $$->Version($2); }
;

RPCProgramName: NAME '=' STRING ';'   { $$ = $3; };

RPCProgramProtocol: PROTOCOL '=' STRING ';'
{
  $$ = IPPROTO_TCP;
  if (! $3->empty()) {
    auto  protoEntry = getprotobyname($3->c_str());
    if (protoEntry) {
      $$ = protoEntry->p_proto;
    }
  }  
  delete $3;
};

RPCProgramVersion: VERSION '=' INTEGER ';'   { $$ = $3;  };

SMTPStanza: SMTP '=' '[' TCP4Ports ']' ';'   { $$ = $4; };

SMTP6Stanza: SMTP6 '=' '[' TCP4Ports ']' ';'   { $$ = $4; };

UPSStanza: UPS '=' '[' VectorOfString ']' ';'    { $$ = $4; };

DNSStanza: DNS '=' '{' DNSResolve '}' ';'
{
  $$ = $4;
};

DNSResolve: RESOLVE '=' '[' DNSQuestions ']' ';'
{
  $$ = $4;
};

DNSQuestions : DNSQuestion
{
  $$ = new vector<Dwm::Dns::MessageQuestion>();
  $$->push_back(*$1);
  delete $1;
}
| DNSQuestions ',' DNSQuestion
{
  $$->push_back(*$3);
  delete $3;
};

DNSQuestion : '{' DNSQuestionData '}'
{
  $$ = $2;
}

DNSQuestionData: DNSQuestionQType
{
  $$ = new Dwm::Dns::MessageQuestion();
  $$->QClass(Dwm::Dns::MessageQuestion::k_classIN);
  //  case-insensitive search our supported question types
  auto  it =
    find_if(supportedQTypes.begin(), supportedQTypes.end(),
            [&] (const pair<string,uint16_t> & p)
            { return std::equal(p.first.begin(), p.first.end(),
                                $1->begin(), $1->end(),
                                [] (const char & c1, const char & c2)
                                { return (std::toupper(c1)
                                          == std::toupper(c2)); }); });
  delete $1;
  if (it != supportedQTypes.end()) {
    $$->QType(it->second);
  }
  else {
    mcrovercfgerror("unsupported DNS qtype");
    return 1;
  }
}
| DNSQuestionQName
{
  $$ = new Dwm::Dns::MessageQuestion();
  $$->QClass(Dwm::Dns::MessageQuestion::k_classIN);
  if (inet_addr($1->c_str()) != INADDR_NONE) {
    string  arpaName;
    Dwm::Dns::ToArpa(*$1, arpaName);
    $$->QName(arpaName);
    $$->QType(Dwm::Dns::MessageQuestion::k_typePTR);
  }
  else {
    $$->QName(*$1);
    $$->QType(Dwm::Dns::MessageQuestion::k_typeA);
  }
  delete $1;
}
| DNSQuestionData DNSQuestionQType {
  auto  it =
    find_if(supportedQTypes.begin(), supportedQTypes.end(),
            [&] (const pair<string,uint16_t> & p)
            { return std::equal(p.first.begin(), p.first.end(),
                                $2->begin(), $2->end(),
                                [] (const char & c1, const char & c2)
                                { return (std::toupper(c1)
                                          == std::toupper(c2)); }); });
  delete $2;
  if (it != supportedQTypes.end()) {
    $$->QType(it->second);
  }
  else {
    mcrovercfgerror("unsupported DNS qtype");
    return 1;
  }
}
| DNSQuestionData DNSQuestionQName {
  if ($$->QType() == Dwm::Dns::MessageQuestion::k_typePTR) {
    if (inet_addr($2->c_str()) != INADDR_NONE) {
      string  arpaName;
      Dwm::Dns::ToArpa(*$2, arpaName);
      $$->QName(arpaName);
    }
    else {
      mcrovercfgerror("DNS PTR qname must be IP address");
      delete $2;
      return 1;
    }
  }
  else {
    $$->QName(*$2);
  }
  delete $2;
};

DNSQuestionQType: QTYPE '=' STRING ';'
{
  $$ = $3;
};

DNSQuestionQName: QNAME '=' STRING ';'
{
  $$ = $3;
};

WebAppStanza: WEBAPP '=' '[' WebAppTargets ']' ';'
{
  $$ = $4;
};

WebAppTargets: WebAppTarget
{
  $$ = new vector<Dwm::Mcrover::WebAppTarget>();
  $$->push_back(*$1);
  delete $1;
}
| WebAppTargets ',' WebAppTarget
{
  $$->push_back(*$3);
  delete $3;
};

WebAppTarget: '{' WebAppTargetParts '}'
{
  $$ = $2;
};

WebAppTargetParts: Uri
{
  $$ = new Dwm::Mcrover::WebAppTarget();
  $$->URI(*$1);
  delete $1;
}
| AppName
{
  $$ = new Dwm::Mcrover::WebAppTarget();
  $$->AppName(*$1);
  delete $1;
}
| Xpath
{
  $$ = new Dwm::Mcrover::WebAppTarget();
  $$->Xpath(*$1);
  delete $1;
}
| AppParams
{
  $$ = new Dwm::Mcrover::WebAppTarget();
  $$->Params(*$1);
  delete $1;
}
| ValidateCertificate
{
  $$ = new Dwm::Mcrover::WebAppTarget();
  $$->ValidateCertificate($1);
}
| WebAppTargetParts Uri        { $$->URI(*$2);  delete $2; }
| WebAppTargetParts AppName    { $$->AppName(*$2); delete $2; }
| WebAppTargetParts Xpath      { $$->Xpath(*$2); delete $2; }
| WebAppTargetParts AppParams  { $$->Params(*$2); delete $2; }
| WebAppTargetParts ValidateCertificate { $$->ValidateCertificate($2); }
;

CredenceStanza: CREDENCE '=' '[' CredenceServiceTargets ']' ';'
{
  $$ = $4;
};

CredenceServiceTargets: CredenceServiceTarget
{
  $$ = new vector<Dwm::Mcrover::TcpDstAddr>();
  $$->push_back(*$1);
  delete $1;
}
| CredenceServiceTargets ',' CredenceServiceTarget
{
  $$->push_back(*$3);
  delete $3;
}
;

CredenceServiceTarget: '{' CredenceServiceTargetParts '}'
{
  $$ = $2;
};

CredenceServiceTargetParts: PORT '=' TCP4Port ';'
{
  $$ = new Dwm::Mcrover::TcpDstAddr();
  $$->Port($3);
}
| NAME '=' STRING ';'
{
  $$ = new Dwm::Mcrover::TcpDstAddr();
  $$->Name(*$3);
  delete $3;
}
| ADDRESS '=' STRING ';'
{
  $$ = new Dwm::Mcrover::TcpDstAddr();
  $$->Address(Dwm::IpAddress(*$3));
  delete $3;
}
| CredenceServiceTargetParts PORT '=' TCP4Port ';'
{
  $$->Port($4);
}
| CredenceServiceTargetParts NAME '=' STRING ';'
{
  $$->Name(*$4);
  delete $4;
}
| CredenceServiceTargetParts ADDRESS '=' STRING ';'
{
  $$->Address(Dwm::IpAddress(*$4));
  delete $4;
}
;


WebStanza: WEB '=' '[' WebTargets ']' ';'
{
  $$ = $4;
};

WebTargets: WebTarget
{
  $$ = new vector<Dwm::Mcrover::WebTarget>();
  $$->push_back(*$1);
  delete $1;
}
| WebTargets ',' WebTarget
{
  $$->push_back(*$3);
  delete $3;
};

WebTarget: '{' WebTargetParts '}'
{
  $$ = $2;
};

WebTargetParts: Uri
{
  $$ = new Dwm::Mcrover::WebTarget();
  $$->URI(*$1);
  $$->GoodResponses().AddRange(make_pair(200,200));
  delete $1;
}
| WebResponses
{
  $$ = new Dwm::Mcrover::WebTarget();
  $$->GoodResponses(*$1);
  delete $1;
}
| WebTargetParts Uri
{
  $$->URI(*$2);
  delete $2;
}
| WebTargetParts WebResponses
{
  $$->GoodResponses(*$2);
  delete $2;
}
;

WebResponses: STATUS '=' IntegerRanges ';'
{
  $$ = $3;
};

IntegerRanges: '[' IntegerRangeSet ']'
{
  $$ = $2;
};

IntegerRangeSet: IntegerRangePair
{
  $$ = new Dwm::Mcrover::RangeSetPair<uint16_t>();
  $$->AddRange(*$1);
  delete $1;
}
| IntegerRangeSet ',' IntegerRangePair
{
  $$->AddRange(*$3);
  delete $3;
};

IntegerRangePair: INTEGER '-' INTEGER
{
  $$ = new pair<uint16_t,uint16_t>($1, $3);
}
| INTEGER
{
  $$ = new pair<uint16_t,uint16_t>($1, $1);
};

Uri: URI '=' STRING ';'
{
  $$ = $3;
};

AppName: APPNAME '=' STRING ';'
{
  $$ = $3;
};

Xpath: XPATH '=' STRING ';'
{
  $$ = $3;
};

AppParams: PARAMS '=' '[' VectorOfString ']' ';'
{
  $$ = $4;
};

ValidateCertificate: VALIDATECERT '=' TRUESTRING ';'
{
  $$ = true;
}
| VALIDATECERT '=' FALSESTRING ';'
{
  $$ = false;
};

Weather: WEATHER '{' WeatherHost '}' ';'
{
  g_config->Weather(*$3);
  delete $3;
};

WeatherHost: HOST '=' STRING ',' PORT '=' TCP4Port
{
  $$ = new Dwm::Mcrover::WeatherConfig();
  $$->Host(*$3);
  $$->Port($7);
  delete $3;
}
| PORT '=' TCP4Port ',' HOST '=' STRING
{
  $$ = new Dwm::Mcrover::WeatherConfig();
  $$->Host(*$7);
  $$->Port($3);
  delete $7;
}
| HOST '=' STRING
{
  $$ = new Dwm::Mcrover::WeatherConfig();
  $$->Host(*$3);
  delete $3;
}
;

VectorOfString: STRING
{
  $$ = new vector<string>();
  $$->push_back(*$1);
  delete $1;
}
| VectorOfString ',' STRING    { $$->push_back(*$3); delete $3; }
;

%%

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/src/DwmMcroverConfigParse.y 12283 $");

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Config::Config()
        : _alertFile("/tmp/alerts"), _runService(false), _service(),
          _syslogFacility("daemon"), _syslogLevel("info"),
          _logLocations(false), _myPack(), _otherPacks(), _servers()
    {
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Config::Parse(const string & path)
    {
      using  batcp = boost::asio::ip::tcp;
        
      bool  rc = false;
      Clear();
      
      mcrovercfgin = fopen(path.c_str(), "r");
      if (mcrovercfgin) {
        g_configPath = path;
        g_config = this;
        rc = (0 == mcrovercfgparse());
        fclose(mcrovercfgin);
        mcrovercfglex_destroy();
      }
      else {
        Syslog(LOG_ERR, "Failed to open config file '%s'", path.c_str());
      }
      if (rc) {
        Syslog(LOG_INFO, "%lu pack members, %lu target servers",
               _myPack.Members().size(), _servers.size());
        if (_service.Addresses().empty()) {
          _service.AddAddress(batcp::endpoint(batcp::v6(), 2123));
          _service.AddAddress(batcp::endpoint(batcp::v4(), 2123));
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & Config::AlertFile() const
    {
      return _alertFile;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & Config::AlertFile(const string & alertFile)
    {
      _alertFile = alertFile;
      return _alertFile;
    }

    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    const string & Config::SyslogFacility() const
    {
      return _syslogFacility;
    }
    
    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    const string & Config::SyslogFacility(const string & facility)
    {
      _syslogFacility = facility;
      return _syslogFacility;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & Config::SyslogLevel() const
    {
      return _syslogLevel;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & Config::SyslogLevel(const string & level)
    {
      _syslogLevel = level;
      return _syslogLevel;
    }

    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    bool Config::SyslogLocations() const
    {
      return _logLocations;
    }
    

    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    bool Config::SyslogLocations(bool logLocations)
    {
      _logLocations = logLocations;
      return _logLocations;
    }

    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    bool Config::RunService() const
    {
      return _runService;
    }
    
    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    bool Config::RunService(bool runService)
    {
      _runService = runService;
      return _runService;
    }

    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    const ServiceConfig & Config::Service() const
    {
      return _service;
    }
    
    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    const ServiceConfig Config::Service(const ServiceConfig & service)
    {
      _service = service;
      return _service;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const LocalHostConfig & Config::Local() const
    {
      return _local;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    LocalHostConfig & Config::Local()
    {
      return _local;
    }

    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    const PackConfig & Config::MyPack() const
    {
      return _myPack;
    }
    
    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    PackConfig & Config::MyPack()
    {
      return _myPack;
    }
    
    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    const PackConfig & Config::MyPack(const PackConfig & packConfig)
    {
        _myPack = packConfig;
        return _myPack;
    }

    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
        const map<string,PackConfig> & Config::OtherPacks() const
    {
      return _otherPacks;
    }

    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    map<string,PackConfig> & Config::OtherPacks()
    {
      return _otherPacks;
    }

    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    AlertOrigin Config::MyOrigin() const
    {
        AlertOrigin  rc;
        rc.PackName(_myPack.PackName());
        rc.MemberAddress(Utils::ThisHostAddr());
        return rc;
    }
    
    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    const vector<TargetHostConfig> & Config::Servers() const
    {
      return _servers;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<TargetHostConfig> &
    Config::Servers(const vector<TargetHostConfig> & servers)
    {
      _servers = servers;
      return _servers;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const WeatherConfig & Config::Weather() const
    {
      return _weather;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const WeatherConfig & Config::Weather(const WeatherConfig & weather)
    {
      _weather = weather;
      return _weather;
    }
    
    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    void Config::Clear()
    {
      _alertFile.clear();
      _runService = false;
      _service.Clear();
      _syslogFacility = "daemon";
      _syslogLevel = "info";
      _logLocations = false;
      _local.Zpools().clear();
      _local.Filesystems().clear();
      _local.Routes().clear();
      _local.Guests().clear();
      _local.Bandits().clear();
      _myPack.Clear();
      _otherPacks.clear();
      _servers.clear();
      _weather = WeatherConfig();
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PrintOtherPacks(std::ostream & os,
                    const std::map<std::string,PackConfig> & otherPacks)
    {
      if (! otherPacks.empty()) {
        auto  iter = otherPacks.begin();
        os << "otherpacks = [\n"
           << "  { name = \"" << iter->first << "\";\n";
        PackConfig::PrintMembers(os, 4, iter->second.Members());
        os << "  }";
        ++iter;
        for (; iter != otherPacks.end(); ++iter) {
          os << ",\n  { name = \"" << iter->first << "\";\n";
          PackConfig::PrintMembers(os, 4, iter->second.Members());
          os << "  }";
        }
        os << "\n];\n\n";
      }
      return;
    }
    
    //-----------------------------------------------------------------------
    //!  
    //-----------------------------------------------------------------------
    std::ostream & operator << (std::ostream & os, const Config & cfg)
    {
      os << "#=============================================================="
        "==============\n"
         << "#  Alert configuration.\n"
         << "#=============================================================="
        "==============\n"
         << "alerts {\n"
         << "    #----------------------------------------------------------"
        "--------------\n"
         << "    #  File in which to persist current alerts.\n"
         << "    #----------------------------------------------------------"
        "--------------\n"
         << "    store = \"" << cfg._alertFile << "\";\n"
         << "};\n\n";
      
      os << cfg._service << '\n';

      os << "#=============================================================="
          "==============\n"
         << "#  syslog configuration.\n"
         << "#=============================================================="
          "==============\n"
         << "syslog {\n"
         << "    #----------------------------------------------------------"
          "--------------\n"
         << "    #  Syslog facility.  Defaults to \"daemon\" if not set.\n"
         << "    #----------------------------------------------------------"
          "--------------\n"
         << "    facility = \"" << cfg._syslogFacility << "\";\n\n"
         << "    #----------------------------------------------------------"
          "--------------\n"
         << "    #  Minimum syslog priority to log.  Defaults to \"info\" if"
          " not set.\n"
         << "    #----------------------------------------------------------"
          "--------------\n"
         << "    level = \"" << cfg._syslogLevel << "\";\n\n"
         << "    #----------------------------------------------------------"
          "--------------\n"
         << "    #  Set to \"yes\" to get {filename:line} in syslog.\n"
         << "    #----------------------------------------------------------"
          "--------------\n"
         << "    logLocations = \"" << (cfg._logLocations ? "yes" : "no")
         << "\";\n"
         << "};\n\n";

      os << cfg._myPack << '\n';
      PrintOtherPacks(os, cfg._otherPacks);
      os << cfg._local << '\n';
      if (! cfg._servers.empty()) {
        os << "#============================================================"
          "================\n"
           << "#  All the network servers I want to check.\n"
           << "#============================================================"
              "================\n"
           << "servers {\n";
        for (const auto & server : cfg._servers) {
          os << server << '\n';
        }
        os << "};\n";
      }
      os << "weather { host = \"" << cfg._weather.Host() << "\", port = "
         << cfg._weather.Port() << " };\n";
      
      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json Config::ToJson() const
    {
      nlohmann::json  j;
      j["alertFile"] = _alertFile;
      j["runService"] = _runService;
      j["service"] = _service.ToJson();
      j["syslogFacility"] = _syslogFacility;
      j["syslogLevel"] = _syslogLevel;
      j["logLocations"] = _logLocations;
      j["local"] = _local.ToJson();
      j["myPack"] = _myPack.ToJson();
      for (auto op : _otherPacks) {
        j["otherPacks"][op.first] = op.second.ToJson();
      }
      for (size_t i = 0; i < _servers.size(); ++i) {
        j["servers"][i] = _servers[i].ToJson();
      }
      if (! _weather.Host().empty()) {
          j["weather"] = _weather.ToJson();
      }
      return j;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Config::FromJson(const nlohmann::json & j)
    {
      bool  rc = false;
      if (j.is_object()) {
        rc = true;
        auto  it = j.find("alertFile");
        if (it != j.end()) {
          if (it->is_string()) {
            _alertFile = it->get<string>();
          }
          else {
            rc = false;
            goto done;
          }
        }
        it = j.find("runService");
        if (it != j.end()) {
          if (it->is_boolean()) {
            _runService = it->get<bool>();
          }
          else {
            rc = false;
            goto done;
          }
        }
          
        it = j.find("service");
        if (it != j.end()) {
          if (! _service.FromJson(*it)) {
            rc = false;
            goto done;
          }
        }
        it = j.find("syslogFacility");
        if (it != j.end()) {
          if (it->is_string()) {
            _syslogFacility = it->get<string>();
          }
          else {
            rc = false;
            goto done;
          }
        }
        it = j.find("syslogLevel");
        if (it != j.end()) {
          if (it->is_string()) {
            _syslogLevel = it->get<string>();
          }
          else {
            rc = false;
            goto done;
          }
        }
        it = j.find("logLocations");
        if (it != j.end()) {
          if (it->is_boolean()) {
            _logLocations = it->get<bool>();
          }
          else {
            rc = false;
            goto done;
          }
        }
        it = j.find("local");
        if (it != j.end()) {
          if (! _local.FromJson(*it)) {
            rc = false;
            goto done;
          }
        }
        it = j.find("myPack");
        if (it != j.end()) {
          if (! _myPack.FromJson(*it)) {
            rc = false;
            goto done;
          }
        }
        it = j.find("otherPacks");
        if (it != j.end()) {
          if (it->is_object()) {
            auto  opit = it->begin();
            for ( ; opit != it->end(); ++opit) {
              PackConfig  packConfig;
              if (packConfig.FromJson(opit.value())) {
                _otherPacks[opit.key()] = packConfig;
              }
              else {
                break;
              }
            }
            if (it->end() != opit) {
              rc = false;
              goto done;
            }
          }
          else {
            rc = false;
            goto done;
          }
        }
        it = j.find("servers");
        if (it != j.end()) {
          if (it->is_array()) {
            size_t  i = 0;
            for ( ; i < it->size(); ++i) {
              TargetHostConfig  thc;
              if (! thc.FromJson((*it)[i])) {
                break;
              }
              _servers.push_back(thc);
            }
            if (it->size() != i) {
              rc = false;
              goto done;
            }
          }
          else {
            rc = false;
            goto done;
          }
        }
        it = j.find("weather");
        if (it != j.end()) {
          if (! _weather.FromJson(*it)) {
            rc = false;
            goto done;
          }
        }
      }
    done:
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & Config::Write(std::ostream & os) const
    {
      std::ostringstream  oss;
      oss << ToJson().dump() << '\n';
      StreamIO::Write(os, oss.str());
      return os;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & Config::Read(std::istream & is)
    {
      string  s;
      if (StreamIO::Read(is, s)) {
        nlohmann::json  j = nlohmann::json::parse(s, nullptr, false);
        if (! j.is_discarded()) {
          if (! FromJson(j)) {
            is.setstate(std::ios_base::badbit);
          }
        }
        else {
          is.setstate(std::ios_base::badbit);
        }
      }
      return is;
    }

  }  // namespace Mcrover

}  // namespace Dwm
