//===========================================================================
// @(#) $DwmPath$
// @(#) $Id$
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2024
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverBackupUtils.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::BackupUtils class implementation
//---------------------------------------------------------------------------

#include <cstdio>
#include <nlohmann/json.hpp>

#include "DwmSvnTag.hh"
#include "DwmIpv4Address.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverBackupAlert.hh"
#include "DwmMcroverBackupUtils.hh"
#include "DwmMcroverDumpDates.hh"
#include "DwmMcroverUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath$");

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool BackupUtils::GetAlerts(const AlertOrigin & origin,
                                const LocalHostConfig & local,
                                AlertBowl & alerts, bool clearAtEntry)
    {
      if (clearAtEntry) {
        alerts.Clear();
      }
      DumpDates    dumpDates;
      Ipv4Address  localAddr(Utils::ThisHostIpv4Addr());
      time_t       now = time((time_t *)0);
      for (const auto dump : local.Dumps()) {
        auto  lastDump = dumpDates.LastDump(dump.first);
        if (now >= (lastDump.second + dump.second)) {
          Syslog(LOG_ERR, "%s not backed up for more than %llu seconds",
                 dump.first.c_str(), dump.second);
          alerts.Add(origin, BackupAlert(localAddr, dump.first.c_str(),
                                         dump.second), now);
        }
        else {
          Syslog(LOG_INFO, "%s backed up %lld seconds ago",
                 dump.first.c_str(), now - lastDump.second);
        }
      }
      return (! alerts.Empty());
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm
