//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/src/DwmMcroverBackupAlert.cc 12329 $
// @(#) $Id: DwmMcroverBackupAlert.cc 12329 2024-01-21 06:43:01Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2023, 2024
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverBackupAlert.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::BackupAlert class implementation
//---------------------------------------------------------------------------

#include <iomanip>

#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmIOUtils.hh"
#include "DwmStreamIO.hh"
#include "DwmIpv4Address.hh"
#include "DwmSvnTag.hh"
#include "DwmMcroverBackupAlert.hh"
#include "DwmMcroverUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/src/DwmMcroverBackupAlert.cc 12329 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    BackupAlert::BackupAlert()
    {
      get<0>(_data) = Ipv4Address(INADDR_NONE);
      get<1>(_data) = "";
      get<2>(_data) = 0;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    BackupAlert::BackupAlert(const Ipv4Address & localAddr,
                             const string & mountPoint)
    {
      get<0>(_data) = localAddr;
      get<1>(_data) = mountPoint;
      get<2>(_data) = 0;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    BackupAlert::BackupAlert(const Ipv4Address & localAddr,
                             const string & mountPoint,
                             uint64_t dumpInterval)
    {
      get<0>(_data) = localAddr;
      get<1>(_data) = mountPoint;
      get<2>(_data) = dumpInterval;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static std::string TimeIntervalString(uint64_t secs)
    {
      std::string  rc;
      uint64_t     secsRemaining = secs;
      uint64_t     days = secsRemaining / (24 * 3600);
      if (days) {
        rc += std::to_string(days) + 'd';
        secsRemaining -= days * (24 * 3600);
      }
      uint64_t  hours = secsRemaining / 3600;
      if (hours) {
        rc += std::to_string(hours) + 'h';
        secsRemaining -= hours * 3600;
      }
      uint64_t  minutes = secsRemaining / 60;
      if (minutes) {
        rc += std::to_string(minutes) + 'm';
        secsRemaining -= minutes * 60;
      }
      if (secsRemaining) {
        rc += std::to_string(secsRemaining) + 's';
      }
      if (rc.empty()) {
        rc = "0s";
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string BackupAlert::DisplayString() const
    {
      return get<1>(_data) + ':' + TimeIntervalString(get<2>(_data))
        + " BACKUP";
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json BackupAlert::ToJson() const
    {
      nlohmann::json  j;
      j["local"] = (string)get<0>(_data);
      j["mount"] = get<1>(_data);
      j["lastDump"] = get<2>(_data);
      return j;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool BackupAlert::FromJson(const nlohmann::json & j)
    {
      Clear();
      bool  rc = false;
      if (j.is_object()) {
        auto  local = j.find("local");
        if ((local != j.end()) && local->is_string()) {
          get<0>(_data) = Ipv4Address(local->get<string>());
          auto  pool = j.find("mount");
          if ((pool != j.end()) && pool->is_string()) {
            get<1>(_data) = pool->get<string>();
            auto  lastDumpTime = j.find("lastDump");
            if ((lastDumpTime != j.end()) && lastDumpTime->is_number()) {
              get<2>(_data) = lastDumpTime->get<uint64_t>();
              rc = true;
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    istream & BackupAlert::Read(istream & is)
    {
      Clear();
      return StreamIO::Read(is, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t BackupAlert::Read(int fd)
    {
      Clear();
      return DescriptorIO::Read(fd, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t BackupAlert::Read(FILE * f)
    {
      Clear();
      return FileIO::Read(f, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & BackupAlert::Write(ostream & os) const
    {
      return StreamIO::Write(os, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t BackupAlert::Write(int fd) const
    {
      return DescriptorIO::Write(fd, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t BackupAlert::Write(FILE *f) const
    {
      return FileIO::Write(f, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t BackupAlert::StreamedLength() const
    {
      return IOUtils::StreamedLength(_data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void BackupAlert::Clear()
    {
      get<0>(_data) = INADDR_NONE;
      get<1>(_data).clear();
      get<2>(_data) = 0;
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm
