//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/apps/qmcrover/DwmMcroverStackedWidget.cc 12392 $
// @(#) $Id: DwmMcroverStackedWidget.cc 12392 2024-06-08 08:19:40Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020, 2024
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverStackedWidget.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::StackedWidget class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <signal.h>
}

#include <algorithm>
#include <deque>
#include <iostream>
#include <regex>
#include <vector>

#include "DwmProcessTable.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverStackedWidget.hh"
#include "DwmMcroverUserActivityTracker.hh"
#include "DwmMcroverNWSRadarPageWidget.hh"

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    StackedWidget::StackedWidget(QWidget *parent)
        : QStackedWidget(parent)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void StackedWidget::nextPage()
    {
      auto  ci = currentIndex() + 1;
      if (ci >= count()) {
        ci = 0;
      }
      setCurrentIndex(ci);
      emit pageChanged(widget(ci));
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void StackedWidget::prevPage()
    {
      auto  ci = currentIndex();
      if (ci == 0) {
        ci = count() - 1;
      }
      else {
        --ci;
      }
      setCurrentIndex(ci);
      emit pageChanged(widget(ci));
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void StackedWidget::pageByName(QString pageName)
    {
      QWidget  *c = findChild<QWidget *>(pageName);
      if (c) {
        auto  idx = indexOf(c);
        if ((idx != -1) && (idx != currentIndex())) {
          setCurrentIndex(idx);
          emit pageChanged(c);
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static std::deque<std::string>  sg_pages = {
      "alertPage", "nwsRadarPage", "weatherPage"
    };

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool GetChildren(const Dwm::ProcessTable & ptbl, pid_t pid,
                            std::vector<Dwm::ProcessInfo> & procs)
    {
      auto  match =
        [=] (const auto & p) { return (pid == p.second.ParentId()); };
      auto  it = ptbl.begin();
      while ((it = std::find_if(it, ptbl.end(), match)) != ptbl.end()) {
        procs.emplace_back(it->second);
        GetChildren(ptbl, it->second.Id(), procs);
        ++it;
      }
      return (! procs.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void KillBigQtWebEngines(pid_t pid, uint64_t maxRss)
    {
      static std::regex  rgx(".*QtWebEngineProcess.*",
                             std::regex::ECMAScript|std::regex::optimize);
      Dwm::ProcessTable  procTbl;
      if (Dwm::GetProcessTable(procTbl)) {
        std::vector<Dwm::ProcessInfo>  procs;
        if (GetChildren(procTbl, pid, procs)) {
          for (auto & p : procs) {
            //  The big hog QtWebEngineProcess instances are not direct
            //  children of our process.
            if ((p.ParentId() != pid) && (p.ResidentSetSize() > maxRss)) {
              if (regex_match(p.Command(), rgx)) {
                Syslog(LOG_INFO, "Killing QtWebEngineProcess process for"
                       " excessive RSS of %u bytes", p.ResidentSetSize());
                if (kill(p.Id(), SIGKILL) == 0) {
                  Syslog(LOG_INFO, "Killed QtWebEngineProcess process %u",
                         p.Id());
                }
                else {
                  Syslog(LOG_ERR, "kill(%u,SIGKILL) failed: %m", p.Id());
                }
              }
            }
          }
        }
      }
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void StackedWidget::newAlerts(std::shared_ptr<std::vector<AlertBowl::EntryType>> alerts) 
    {
      if (! alerts->empty()) {
        QWidget  *pw = parentWidget();
        if (pw) {
          pw->setStyleSheet("background-color: #400000;");
        }
      }
      if (UserInactive(30)) {
        std::rotate(sg_pages.begin(), sg_pages.begin() + 1, sg_pages.end());
        if (alerts->empty() && (sg_pages.front() == "alertPage")) {
          std::rotate(sg_pages.begin(), sg_pages.begin() + 1, sg_pages.end());
        }
        pageByName(sg_pages.front().c_str());
      }        
      KillBigQtWebEngines(getpid(), 900000000ull);
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void StackedWidget::noAlerts()
    {
      if (UserInactive(30)) {
        pageByName(sg_pages.front().c_str());
        std::rotate(sg_pages.begin(), sg_pages.begin() + 1,
                    sg_pages.end());
      }
      QWidget  *pw = parentWidget();
      if (pw) {
        pw->setStyleSheet("background-color: #000000;");
      }
      KillBigQtWebEngines(getpid(), 900000000ull);
      return;
    }

  }  // namespace Mcrover

}  // namespace Dwm
