//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/apps/qmcrover/DwmMcroverAlertFetcher.cc 12361 $
// @(#) $Id: DwmMcroverAlertFetcher.cc 12361 2024-05-21 22:32:47Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020, 2022
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverAlertFetcher.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::AlertFetcher class implementation
//---------------------------------------------------------------------------

#include <sstream>

#include "DwmSocket.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverPackMember.hh"
#include "DwmMcroverUtils.hh"
#include "DwmMcroverAlertFetcher.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/apps/qmcrover/DwmMcroverAlertFetcher.cc 12361 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    AlertFetcher::AlertFetcher(const Config & config)
        : QObject(), _config(config), _peer(), _run(false), _runmtx(),
          _runcv(), _thread()
    {
      qRegisterMetaType<std::shared_ptr<std::vector<Dwm::Mcrover::AlertBowl::EntryType>>>("std::shared_ptr<std::vector<Dwm::Mcrover::AlertBowl::EntryType>>");
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    AlertFetcher::~AlertFetcher()
    {
      Stop();
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertFetcher::Start()
    {
      bool  rc = false;
      _run = true;
      try { 
        _thread = std::thread(&AlertFetcher::Run, this);
        rc = true;
      }
      catch (...) {
        Syslog(LOG_ERR, "Exception starting AlertFetcher thread");
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertFetcher::Stop()
    {
      {
        lock_guard<mutex>  lk(_runmtx);
        _run = false;
      }
      _runcv.notify_one();
      if (_thread.joinable()) {
        _thread.join();
        Syslog(LOG_INFO, "AlertFetcher thread joined.");
      }
      return true;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertFetcher::GetMcroverHosts(vector<PackMemberConfig> & hosts) const
    {
      namespace baip = boost::asio::ip;
      using  batcp = baip::tcp;
      
      hosts.clear();
      if (_config.RunService()) {
        //  Local host is running service.  Try there first.
        Mcrover::PackMemberConfig  localAsPackMember;
        localAsPackMember.Name(Mcrover::Utils::ThisHostName());
        baip::address  addr(baip::address::from_string("127.0.0.1"));
        localAsPackMember.AddAddress(batcp::endpoint(addr, 2123));
        for (const auto & servAddr : _config.Service().Addresses()) {
          localAsPackMember.AddAddress(servAddr);
        }
        hosts.push_back(localAsPackMember);
      }
      for (const auto & memberConfig : _config.MyPack().Members()) {
        hosts.push_back(memberConfig.second);
      }
      return !(hosts.empty());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool
    AlertFetcher::GetAlerts(const PackMemberConfig & host,
                            vector<Dwm::Mcrover::AlertBowl::EntryType> & alerts,
                            Dwm::Mcrover::RoverAlert::FailureType & failure)
    {
      bool  rc = false;
      PackMember  packMember(host);
      AlertBowl   alertBowl;
      if (packMember.GetAllAlerts(alertBowl, failure)) {
        alertBowl.Get(alerts);
        std::sort(alerts.begin(), alerts.end(),
                  [] (const auto & alert1, const auto & alert2)
                  { return (alert1.second.Newest().second
                            > alert2.second.Newest().second); });
        rc = true;
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertFetcher::GetAlerts(std::string & chosenHost,
                                 vector<Mcrover::AlertBowl::EntryType> & alerts)
    {
      bool  rc = false;
      chosenHost.clear();
      vector<Mcrover::PackMemberConfig>  hosts;
      if (GetMcroverHosts(hosts)) {
        for (const auto & host : hosts) {
          Mcrover::RoverAlert::FailureType  failure;
          if (GetAlerts(host, alerts, failure)) {
            Syslog(LOG_INFO, "Got %lu alerts from %s", alerts.size(),
                   host.Name().c_str());
            chosenHost = host.Name();
            rc = true;
            break;
          }
          else {
            Syslog(LOG_ERR, "Failed to get alerts from %s",
                   host.Name().c_str());
          }
        }
      }
      else {
        Syslog(LOG_ERR, "No mcrover servers found");
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void AlertFetcher::Run()
    {
      bool    keepRunning = true;
      string  curHost;
      
      Syslog(LOG_INFO, "AlertFetcher thread started");
      while (keepRunning) {
        string  chosenHost;
        std::shared_ptr<std::vector<Dwm::Mcrover::AlertBowl::EntryType>>  alerts
          = make_shared<vector<AlertBowl::EntryType>>();
        if (GetAlerts(chosenHost, *alerts)) {
          Syslog(LOG_INFO, "chose host %s", chosenHost.c_str());
          emit newAlerts(alerts);
          if (chosenHost != curHost) {
            curHost = chosenHost;
            emit newPeer(QString(curHost.c_str()));
          }
          if (alerts->size() == 0) {
            emit noAlerts();
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to get alerts");
        }
        
        {
          unique_lock<mutex>  lk(_runmtx);
          if (_runcv.wait_for(lk, 30s) != std::cv_status::timeout) {
            keepRunning = _run;
          }
        }
      }
      Syslog(LOG_INFO, "AlertFetcher thread done");
      
      return;
    }
    

  }  // namespace Mcrover

}  // namespace Dwm
