//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/apps/mcroverd/DwmMcroverServer.hh 12195 $
// @(#) $Id: DwmMcroverServer.hh 12195 2023-06-26 06:20:30Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverServer.hh
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::Server class declaration
//---------------------------------------------------------------------------

#ifndef _DWMMCROVERSERVER_HH_
#define _DWMMCROVERSERVER_HH_

#include <atomic>
#include <memory>
#include <thread>
#include <boost/asio.hpp>

#include "DwmMcroverConfig.hh"
#include "DwmMcroverResponder.hh"
#include "DwmMcroverUtils.hh"

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  Encapsulates a server to listen for TCP connections from those
    //!  interested in retrieving our alerts.  Spawns a Responder for
    //!  each connection, which is responsible for authentication and
    //!  responding to client requests.
    //------------------------------------------------------------------------
    class Server
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct from a mcroverd configuration @c config.
      //----------------------------------------------------------------------
      Server(const Config & config)
          : _config(config), _keyStash(config.Service().KeyDirectory()),
            _knownKeys(config.Service().KeyDirectory()),
            _allowedClients(config.Service().AllowedClients()),
            _ioContext(), _acceptors(), _responders()
      {
        namespace ip = boost::asio::ip;
        using boost::asio::ip::tcp;

        _origin.PackName(config.MyPack().PackName());
        _origin.MemberAddress(Utils::ThisHostAddr());
        
        for (auto & ep : config.Service().Addresses()) {
          ip::tcp::acceptor  acc(_ioContext.get_executor());
          if (ep.address().is_v6()) {
            acc.open(ip::tcp::v6());
            acc.set_option(ip::v6_only(true));
          }
          else {
            acc.open(ip::tcp::v4());
          }
          boost::asio::socket_base::reuse_address option(true);
          acc.set_option(option);
          acc.bind(ep);
          auto & a = _acceptors.emplace_back(std::move(acc));
          a.listen();
          AcceptLoop(a);
        }
      }

      //----------------------------------------------------------------------
      //!  Returns the key stash.
      //----------------------------------------------------------------------
      const Credence::KeyStash & GetKeyStash() const  { return _keyStash; }
      
      //----------------------------------------------------------------------
      //!  Returns the known keys.
      //----------------------------------------------------------------------
      const Credence::KnownKeys & GetKnownKeys() const  { return _knownKeys; }
      
      //----------------------------------------------------------------------
      //!  Returns the configuration.
      //----------------------------------------------------------------------
      const Config GetConfig() const  { return _config; }

      //----------------------------------------------------------------------
      //!  Returns the path to the alert file.
      //----------------------------------------------------------------------
      const std::string & AlertFile() const;

      //----------------------------------------------------------------------
      //!  Starts the server.  Returns true on success, false on failure.
      //----------------------------------------------------------------------
      bool Start();
      
      //----------------------------------------------------------------------
      //!  Stops the server.
      //----------------------------------------------------------------------
      void Stop();
      
    private:
      Ipv4Address                                 _addr;
      uint16_t                                    _port;
      Config                                      _config;
      Credence::KeyStash                          _keyStash;
      Credence::KnownKeys                         _knownKeys;
      std::set<IpPrefix>                          _allowedClients;
      AlertOrigin                                 _origin;
      boost::asio::io_context                     _ioContext;
      std::deque<boost::asio::ip::tcp::acceptor>  _acceptors;
      std::atomic<bool>                           _run;
      std::thread                                 _thread;
      std::vector<std::shared_ptr<Responder>>     _responders;

      void AcceptLoop(boost::asio::ip::tcp::acceptor & a);
      bool ClientAllowed(const boost::asio::ip::address & epAddr) const;
      void CleanupResponders();
      void Run();
    };
    
  }  // namespace Mcrover

}  // namespace Dwm

#endif  // _DWMMCROVERSERVER_HH_
