//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/apps/mcroverd/DwmMcroverListenSocket.cc 11354 $
// @(#) $Id: DwmMcroverListenSocket.cc 11354 2020-12-08 16:21:12Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverListenSocket.cc
//!  \author Daniel W. McRobb
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <unistd.h>
}

#include <cstring>

#include "DwmSysLogger.hh"
#include "DwmMcroverListenSocket.hh"

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ListenSocket::ListenSocket()
        : _listenAddr(Ipv4Address(INADDR_NONE), 2123), _fd(-1)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ListenSocket::ListenSocket(const TCPAddress & listenAddr)
        : _listenAddr(listenAddr), _fd(-1)
    {
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ListenSocket::Listen()
    {
      bool  rc = false;
      if (Open()) {
        ReuseAddr();
        if (Bind()) {
          if (listen(_fd, 10) == 0) {
            rc = true;
          }
          else {
            Close();
          }
        }
        else {
          Close();
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ListenSocket::Open()
    {
      switch (_listenAddr.Addr().Family()) {
        case AF_INET:
          _fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
          break;
        case AF_INET6:
          _fd = socket(PF_INET6, SOCK_STREAM, IPPROTO_TCP);
          break;
        default:
          break;
      }
      return (0 <= _fd);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    int ListenSocket::Accept(TCPAddress & clientAddr)
    {
      int  rc = -1;
      if (0 <= _fd) {
        struct sockaddr_in   sockAddr;
        struct sockaddr_in6  sockAddr6;
        struct sockaddr      *addrp;
        socklen_t             addrlen;
        switch (_listenAddr.Addr().Family()) {
          case AF_INET6:
            sockAddr6.sin6_family = AF_INET6;
#ifndef __linux__
            sockAddr6.sin6_len = sizeof(sockAddr6);
#endif
            addrp = (struct sockaddr *)&sockAddr6;
            addrlen = sizeof(sockAddr6);
            rc = accept(_fd, addrp, &addrlen);
            clientAddr = TCPAddress(Ipv6Address(sockAddr6.sin6_addr),
                                    ntohs(sockAddr6.sin6_port));
            break;
          case AF_INET:
            sockAddr.sin_family = AF_INET;
#ifndef __linux__
            sockAddr.sin_len = sizeof(sockAddr);
#endif
            addrp = (struct sockaddr *)&sockAddr;
            addrlen = sizeof(sockAddr);
            rc = accept(_fd, addrp, &addrlen);
            clientAddr = TCPAddress(Ipv4Address(sockAddr.sin_addr.s_addr),
                                    ntohs(sockAddr.sin_port));
            break;
          default:
            break;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void ListenSocket::Close()
    {
      if (0 <= _fd) {
        close(_fd);
        _fd = -1;
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    int ListenSocket::Descriptor() const
    {
      return _fd;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ListenSocket::operator < (const ListenSocket & ls) const
    {
      return (_listenAddr < ls._listenAddr);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ListenSocket::operator == (const ListenSocket & ls) const
    {
      return (_listenAddr == ls._listenAddr);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void ListenSocket::ReuseAddr()
    {
      if (0 <= _fd) {
        int  optval = 1;
        socklen_t optlen = sizeof(optval);
        setsockopt(_fd, SOL_SOCKET, SO_REUSEADDR, &optval, optlen);
#ifdef SO_REUSEPORT
        setsockopt(_fd, SOL_SOCKET, SO_REUSEPORT, &optval, optlen);
#endif
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ListenSocket::Bind()
    {
      bool  rc = false;
      if (0 <= _fd) {
        switch (_listenAddr.Addr().Family()) {
          case PF_INET:
            rc = BindIpv4();
            break;
          case PF_INET6:
            rc = BindIpv6();
            break;
          default:
            break;
        }
      }
      
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ListenSocket::BindIpv4()
    {
      bool  rc = false;

      Syslog(LOG_INFO, "Binding %s", ((std::string)_listenAddr).c_str());
      struct sockaddr_in  sockAddr;
      memset(&sockAddr, 0, sizeof(sockAddr));
#ifndef __linux__
      sockAddr.sin_len = sizeof(sockAddr);
#endif
      sockAddr.sin_family = AF_INET;
      sockAddr.sin_addr.s_addr = _listenAddr.Addr().Addr<Ipv4Address>()->Raw();
      sockAddr.sin_port = htons(_listenAddr.Port());
      socklen_t  addrLen = sizeof(sockAddr);
      if (::bind(_fd, (const struct sockaddr *)&sockAddr, addrLen) == 0) {
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ListenSocket::BindIpv6()
    {
      bool  rc = false;
      
      Syslog(LOG_INFO, "Binding %s", ((std::string)_listenAddr).c_str());
      struct sockaddr_in6  sockAddr;
      memset(&sockAddr, 0, sizeof(sockAddr));
#ifndef __linux__
      sockAddr.sin6_len = sizeof(sockAddr);
#endif
      sockAddr.sin6_family = AF_INET6;
      sockAddr.sin6_addr =
        (struct in6_addr)(*(_listenAddr.Addr().Addr<Ipv6Address>()));
      sockAddr.sin6_port = htons(_listenAddr.Port());
      socklen_t  addrLen = sizeof(sockAddr);
      if (::bind(_fd, (const struct sockaddr *)&sockAddr, addrLen) == 0) {
        rc = true;
      }
      return rc;
    }
    
  }  // namespace Mcrover

}  // namespace Dwm
