//===========================================================================
// @(#) $DwmPath$
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2023
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file mcrovercfg.cc
//!  \author Daniel W. McRobb
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <unistd.h>
}

#include <algorithm>
#include <fstream>

#include "DwmDateTime.hh"
#include "DwmSvnTag.hh"
#include "DwmMcroverConfig.hh"
#include "DwmMcroverPackMember.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath$");

using namespace std;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool GetPeerConfig(const Dwm::Mcrover::PackMemberConfig & peerConfig,
                          Dwm::Mcrover::Config & cfg)
{
  Dwm::Mcrover::PackMember  packMember(peerConfig);
  return packMember.GetConfig(cfg);
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool GetMcroverPeers(const Dwm::Mcrover::Config & config,
                            vector<Dwm::Mcrover::PackMemberConfig> & hosts)
{
  hosts.clear();
  for (const auto & memberConfig : config.MyPack().Members()) {
    hosts.push_back(memberConfig.second);
  }
  return (! hosts.empty());
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool GetPeerConfig(const Dwm::Mcrover::Config & mycfg,
                          const string & peerName,
                          Dwm::Mcrover::Config & peerConfig)
{
  bool  rc = false;
  vector<Dwm::Mcrover::PackMemberConfig>  memberConfigs;
  if (GetMcroverPeers(mycfg, memberConfigs)) {
    auto  it = find_if(memberConfigs.begin(), memberConfigs.end(),
                       [&peerName] (const auto & c)
                       { return (c.Name() == peerName); });
    if (it != memberConfigs.end()) {
      rc = GetPeerConfig(*it, peerConfig);
    }
    else {
      cerr << "pack member '" << peerName << "' not found in config\n";
    }
  }
  else {
    cerr << "unable to find pack members in config\n";
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void Usage(const char *argv0)
{
  std::cerr << "Usage: " << argv0 << " [-j] [-p packMemberName] [configFile]\n";
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  using namespace Dwm;

  bool    emitJson = false;
  string  configFile = "/usr/local/etc/mcroverd.cfg";
  string  packMemberName;
  
  int  optChar;
  while ((optChar = getopt(argc, argv, "jp:")) != -1) {
    switch (optChar) {
      case 'j':
        emitJson = true;
        break;
      case 'p':
        packMemberName = optarg;
        break;
      default:
        Usage(argv[0]);
        return -1;
        break;
    }
  }

  if (optind < argc) {
    configFile = argv[optind];
  }
  int  rc = -1;

  Mcrover::Config  cfg;
  
  if (cfg.Parse(configFile)) {
    if (! packMemberName.empty()) {
      Mcrover::Config  peerConfig;
      if (GetPeerConfig(cfg, packMemberName, peerConfig)) {
        if (emitJson) {
          cout << peerConfig.ToJson().dump(4) << '\n';
        }
        else {
          cout << peerConfig << '\n';
        }
        rc = 0;
      }
    }
    else {
      if (emitJson) {
        cout << cfg.ToJson().dump(4) << '\n';
      }
      else {
        cout << cfg << '\n';
      }
      rc = 0;
    }
  }
  else {
    ifstream  ifs(configFile);
    if (ifs) {
      nlohmann::json  j = j.parse(ifs, nullptr, false);
      if (! j.is_discarded()) {
        if (cfg.FromJson(j)) {
          rc = 0;
          if (emitJson) {
            cout << cfg.ToJson().dump(4) << '\n';
          }
          else {
            cout << cfg;
          }
        }
        else {
          cerr << "Failed to parse '" << configFile << "' as JSON\n";
        }
      }
      ifs.close();
    }
  }
        
  done:
  return rc;
}

