//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPBootstrap.cc 12122 2022-12-11 04:42:58Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPBootstrap.cc
//!  \brief Dwm::RDAP::Bootstram implementation
//---------------------------------------------------------------------------

#include <fstream>
#include <iostream>
#include <iomanip>
#include <string>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmWebUtils.hh"
#include "DwmWebUtilsUrl.hh"
#include "DwmRDAPBootstrap.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/trunk/src/DwmRDAPBootstrap.cc 12122 $");

using namespace std;

namespace Dwm {

  namespace RDAP {

    std::mutex                             Bootstrap::_mtx;
    Ipv4Routes<std::vector<std::string> >  Bootstrap::_addrs2rdaps;

    //------------------------------------------------------------------------
    //!  This is only here to force a default load of the bootstrap.  There
    //!  are no non-static members of the Bootstrap class.
    //------------------------------------------------------------------------
    static Bootstrap  g_bootstrap;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Bootstrap::Bootstrap()
    {
      if (Empty()) {
        Load();
      }
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Bootstrap::Load(const std::string & bootstrapURI)
    {
      {
        lock_guard<mutex>  lock(_mtx);

        _addrs2rdaps.Clear();
        nlohmann::json  jv = GetJsonResponse(bootstrapURI);
        for (int s = 0; s < jv["services"].size(); ++s) {
          for (int n = 0; n < jv["services"][s][0].size(); ++n) {
            for (int rds = 0; rds < jv["services"][s][1].size(); ++rds) {
              vector<string> & e =
                _addrs2rdaps[Ipv4Prefix(jv["services"][s][0][n].get<string>())];
              e.push_back(jv["services"][s][1][rds].get<string>());
              sort(e.begin(), e.end());
            }
          }
        }
      }
      
      return (! _addrs2rdaps.Empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Bootstrap::Empty()
    {
      lock_guard<mutex>  lock(_mtx);
      return _addrs2rdaps.Empty();
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    vector<string> Bootstrap::RDAPServers(const Ipv4Address & addr)
    {
      vector<string>                   rc;
      pair<Ipv4Prefix,vector<string>>  result;
      lock_guard<mutex>                lock(_mtx);
      if (_addrs2rdaps.FindLongest(addr, result)) {
        rc = result.second;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Ipv4Prefix Bootstrap::MatchingPrefix(const Ipv4Address & addr)
    {
      Ipv4Prefix  rc("0/0");
      pair<Ipv4Prefix,vector<string>>  result;
      lock_guard<mutex>                lock(_mtx);
      if (_addrs2rdaps.FindLongest(addr, result)) {
        rc = result.first;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json Bootstrap::GetJsonResponse(const std::string & uristr)
    {
      nlohmann::json  rc;
      WebUtils::Url   url;
      if (url.Parse(uristr)) {
        if (url.Scheme() == "file") {
          std::string  path(url.Path());
          ifstream     is(path.c_str());
          if (is) {
            rc = nlohmann::json::parse(is, nullptr, false);
          }
          return rc;
        }
        else if ((url.Scheme() == "https")
                 || (url.Scheme() == "http")) {
          WebUtils::GetJson(uristr, rc);
        }
      }
      else {
        std::ifstream  is(uristr);
        if (is) {
          rc = nlohmann::json::parse(is, nullptr, false);
        }
        else {
          Syslog(LOG_ERR, "Failed to open '%s'", uristr.c_str());
        }
      }
      return rc;
    }

  }  // namespace RDAP

}  // namespace Dwm
