//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmCountryCode.hh 12073 2022-11-24 05:17:58Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmCountryCode.hh
//!  \brief Dwm::CountryCode class definition
//---------------------------------------------------------------------------

#ifndef _DWMCOUNTRYCODE_HH_
#define _DWMCOUNTRYCODE_HH_

#include <string>
#include <tuple>

#include <nlohmann/json.hpp>

#include "DwmDescriptorIOCapable.hh"
#include "DwmFileIOCapable.hh"
#include "DwmStreamIOCapable.hh"
#include "DwmStreamedLengthCapable.hh"

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  Encapsulates country code information.  2-letter code, 3-letter code
  //!  and full country name.
  //--------------------------------------------------------------------------
  class CountryCode
    : public Dwm::DescriptorIOCapable, public Dwm::FileIOCapable,
      public Dwm::StreamIOCapable, public Dwm::StreamedLengthCapable
  {
  public:
    //------------------------------------------------------------------------
    //!  Default constructor
    //------------------------------------------------------------------------
    CountryCode() = default;

    //------------------------------------------------------------------------
    //!  Construct from a JSON value.
    //------------------------------------------------------------------------
    CountryCode(const nlohmann::json & jv);

    //------------------------------------------------------------------------
    //!  Return the country code as a JSON value.
    //------------------------------------------------------------------------
    nlohmann::json Json() const;
    
    //------------------------------------------------------------------------
    //!  Returns the 2-letter country code.
    //------------------------------------------------------------------------
    const std::string & Alpha2() const;
    
    //------------------------------------------------------------------------
    //!  Sets and returns the 2-letter country code.  Will assert() that
    //!  @c alpha2 is 2 characters long.
    //------------------------------------------------------------------------
    const std::string & Alpha2(const std::string & alpha2);

    //------------------------------------------------------------------------
    //!  Returns the 3-letter country code.
    //------------------------------------------------------------------------
    const std::string & Alpha3() const;
    
    //------------------------------------------------------------------------
    //!  Sets and returns the 3-letter country code.  Will assert that
    //!  @c alpha3 is 3 characters long.
    //------------------------------------------------------------------------
    const std::string & Alpha3(const std::string & alpha3);
    
    //------------------------------------------------------------------------
    //!  Returns the full country name.
    //------------------------------------------------------------------------
    const std::string & Name() const;
    
    //------------------------------------------------------------------------
    //!  Sets and returns the full country name.
    //------------------------------------------------------------------------
    const std::string & Name(const std::string & name);

    //------------------------------------------------------------------------
    //!  Reads the country code from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    std::istream & Read(std::istream & is) override;
    
    //------------------------------------------------------------------------
    //!  Reads the country code from a descriptor.  Returns the number of
    //!  bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    ssize_t Read(int fd) override;
    
    //------------------------------------------------------------------------
    //!  Reads the country code from a FILE.  Returns 1 on success,
    //!  0 on failure (fread() semantics).
    //------------------------------------------------------------------------
    size_t Read(FILE *f) override;
    
    //------------------------------------------------------------------------
    //!  Writes the country code to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    std::ostream & Write(std::ostream & os) const override;
    
    //------------------------------------------------------------------------
    //!  Writes the country code to a file descriptor.  Returns the number of
    //!  bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    ssize_t Write(int fd) const override;
    
    //------------------------------------------------------------------------
    //!  Writes the country code to a FILE.  Returns 1 on success, 0 on
    //!  failure (fwrite() semantics).
    //------------------------------------------------------------------------
    size_t Write(FILE *f) const override;
    
    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if one of the
    //!  Write() members is called.
    //------------------------------------------------------------------------
    uint64_t StreamedLength() const override;

    //------------------------------------------------------------------------
    //!  operator ==
    //------------------------------------------------------------------------
    bool operator == (const CountryCode & cc) const
    {
      return (_data == cc._data);
    }
    
  private:
    std::tuple<std::string,std::string,std::string>  _data;
  };
  
}  // namespace Dwm

#endif  // _DWMCOUNTRYCODE_HH_
