//===========================================================================
// @(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/src/DwmPiRotaryEncoderReader.cc 8696 $
// @(#) $Id: DwmPiRotaryEncoderReader.cc 8696 2016-07-08 03:18:48Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPiRotaryEncoderReader.cc
//!  \brief Dwm::Pi::RotaryEncoder class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/event.h>
  #include <sys/time.h>
  #include <sys/ioctl.h>
  #include <unistd.h>
  #include <pthread_np.h>
  #include <sys/gpiorotenc.h>
  #include <fcntl.h>
}

#include <cassert>
#include <iostream>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmPiRotaryEncoderReader.hh"

static const Dwm::SvnTag  svntag("@(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/src/DwmPiRotaryEncoderReader.cc 8696 $");

using namespace std;

namespace Dwm {

  namespace Pi {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RotaryEncoderReader::RotaryEncoderReader(const std::string & device)
        : _device(device), _devfd(-1), _stateMutex(),
          _currentState(&RotaryEncoderReader::Initial), _watchersMutex(),
          _watchers(), _thread(), _run(false)
    {
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RotaryEncoderReader::~RotaryEncoderReader()
    {
      Stop();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & RotaryEncoderReader::DeviceName() const
    {
      return _device;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RotaryEncoderReader::StateName(StateFunc state) const
    {
      static const string stateNames[] = {
        "Initial",
        "00",
        "10",
        "11",
        "01",
        "Unknown"
      };
      if (state == &RotaryEncoderReader::Initial) {
        return stateNames[0];
      }
      else if (state == &RotaryEncoderReader::State00) {
        return stateNames[1];
      }
      else if (state == &RotaryEncoderReader::State10) {
        return stateNames[2];
      }
      else if (state == &RotaryEncoderReader::State11) {
        return stateNames[3];
      }
      else if (state == &RotaryEncoderReader::State01) {
        return stateNames[4];
      }
      else {
        return stateNames[5];
      }
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoderReader::Initial(uint8_t ab)
    {
      lock_guard<mutex>  lock(_stateMutex);
      switch (ab) {
        case 0:
          _currentState = &RotaryEncoderReader::State00;
          break;
        case 1:
          _currentState = &RotaryEncoderReader::State01;
          break;
        case 2:
          _currentState = &RotaryEncoderReader::State10;
          break;
        case 3:
          _currentState = &RotaryEncoderReader::State11;
          break;
        default:
          break;
      }
      Syslog(LOG_INFO, "RotaryEncoderReader(%s) Initial -> %s",
             _device.c_str(), StateName(_currentState).c_str());
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoderReader::State00(uint8_t ab)
    {
      if (ab == 0x2) {
        ChangeState(&RotaryEncoderReader::State10, true);
      }
      else if (ab == 0x1) {
        ChangeState(&RotaryEncoderReader::State01, false);
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoderReader::State10(uint8_t ab)
    {
      if (ab == 0x0) {
        ChangeState(&RotaryEncoderReader::State00, false);
      }
      else if (ab == 0x3) {
        ChangeState(&RotaryEncoderReader::State11, true);
      }
      return;
    }
        
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoderReader::State11(uint8_t ab)
    {
      if (ab == 0x1) {
        ChangeState(&RotaryEncoderReader::State01, true);
      }
      else if (ab == 0x2) {
        ChangeState(&RotaryEncoderReader::State10, false);
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoderReader::State01(uint8_t ab)
    {
      if (ab == 0x0) {
        ChangeState(&RotaryEncoderReader::State00, true);
      }
      else if (ab == 0x3) {
        ChangeState(&RotaryEncoderReader::State11, false);
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoderReader::AddWatcher(RotaryEncoderReaderWatcher *watcher)
    {
      lock_guard<mutex>  lock(_watchersMutex);
      _watchers.insert(watcher);
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void
    RotaryEncoderReader::RemoveWatcher(RotaryEncoderReaderWatcher *watcher)
    {
      lock_guard<mutex>  lock(_watchersMutex);
      auto  it = _watchers.find(watcher);
      if (it != _watchers.end()) {
        _watchers.erase(it);
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RotaryEncoderReader::Start(uint16_t readsPerSecond)
    {
      bool  rc = false;
      _devfd = open((string("/dev/") + _device).c_str(), O_RDONLY);
      if (_devfd >= 0) {
        //  Start the pin polling thread
        _run = true;
        _thread = thread(&RotaryEncoderReader::Run, this);
        pthread_set_name_np(_thread.native_handle(), "encoder reader");
        rc = true;
        Syslog(LOG_INFO, "RotaryEncoderReader(%s) started", _device.c_str());
      }
      else {
        Syslog(LOG_ERR, "Failed to open /dev/%s: %m", _device.c_str());
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoderReader::Stop()
    {
      _run = false;
      if (_thread.joinable()) {
        _thread.join();
        close(_devfd);
        _devfd = -1;
        Syslog(LOG_INFO, "RotaryEncoderReader(%s) stopped", _device.c_str());
      }
      lock_guard<mutex>  lock(_stateMutex);
      _currentState = &RotaryEncoderReader::Initial;
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void
    RotaryEncoderReader::ChangeState(RotaryEncoderReader::StateFunc state,
                                     bool clockwise)
    {
      lock_guard<mutex>  lock(_stateMutex);
      if (_currentState != state) {
        if (SysLogger::MinimumPriority() >= LOG_DEBUG) {
          Syslog(LOG_DEBUG, "RotaryEncoderReader(%s) state %s -> %s",
                 _device.c_str(), StateName(_currentState).c_str(),
                 StateName(state).c_str());
        }
        _currentState = state;
        if (clockwise) {
          EmitClockwiseEvent();
        }
        else {
          EmitCounterClockwiseEvent();
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoderReader::EmitClockwiseEvent()
    {
      lock_guard<mutex>   lock(_watchersMutex);
      if (! _watchers.empty()) {
        for (auto watcher : _watchers) {
          watcher->EncoderMovedClockwise();
        }
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoderReader::EmitCounterClockwiseEvent()
    {
      lock_guard<mutex>   lock(_watchersMutex);
      if (! _watchers.empty()) {
        for (auto watcher : _watchers) {
          watcher->EncoderMovedCounterClockwise();
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoderReader::Run()
    {
      uint8_t  ab;
      int      val;
      if (ioctl(_devfd, ROTENCGET, &val) == 0) {
        ab = val & 0xFF;
        (this->*_currentState)(ab);
      }
      else {
        Syslog(LOG_ERR, "ioctl(%d,ROTENCGET,%p) failed!", _devfd, &val);
      }
      
      int  kq = kqueue();
      assert(kq >= 0);

      struct kevent  changeList, eventList;
      EV_SET(&changeList, _devfd, EVFILT_READ, EV_ADD|EV_CLEAR, 0, 0, NULL);
      struct timespec  timeout = { 0, 500000000L };
      
      while (_run) {
        int  numEvents = kevent(kq, &changeList, 1, &eventList, 1, &timeout);
        if (numEvents > 0) {
          //  There are state changes to read.  Read them all and drive our
          //  state machine with them.
          for (int n = 0; n < eventList.data; ++n) {
            if (read(_devfd, &ab, sizeof(ab)) == sizeof(ab)) {
              (this->*_currentState)(ab);
            }
          }
        }
      }
      close(kq);
      
      return;
    }
    
  }  // namespace Pi
  
}  // namespace Dwm
