//===========================================================================
// @(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/src/DwmPiRotaryEncoder.cc 8747 $
// @(#) $Id: DwmPiRotaryEncoder.cc 8747 2016-07-31 03:56:30Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPiRotaryEncoder.cc
//!  \brief RotaryEncoder class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <pthread_np.h>
}

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmPiRotaryEncoder.hh"

static const Dwm::SvnTag  svntag("@(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/src/DwmPiRotaryEncoder.cc 8747 $");

using namespace std;

namespace Dwm {

  namespace Pi {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::EncoderMovedClockwise()
    {
      _moves.PushBack(true);
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::EncoderMovedCounterClockwise()
    {
      _moves.PushBack(false);
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RotaryEncoder::RotaryEncoder(const string & device)
        : _reader(device), _stateMutex(), _state(&RotaryEncoder::Initial),
          _moves(), _counterMutex(), _counter(0), _minObservedCounter(0),
          _maxObservedCounter(0), _thread(), _run(false), _watchers()
    {}

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RotaryEncoder::~RotaryEncoder()
    {
      Stop();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const RotaryEncoderReader & RotaryEncoder::Reader() const
    {
      return _reader;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RotaryEncoder::StateEnum RotaryEncoder::StateValue() const
    {
      StateEnum  rc = e_stateInitial;
      if (_state == &RotaryEncoder::Stopped) {
        rc = e_stateStopped;
      }
      else if (_state == &RotaryEncoder::MovingClockwise) {
        rc = e_stateMovingClockwise;
      }
      else if (_state == &RotaryEncoder::MovingCounterClockwise) {
        rc = e_stateMovingCounterClockwise;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    int64_t RotaryEncoder::Counter() const
    {
      lock_guard<mutex>  lock(_counterMutex);
      return _counter;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    int64_t RotaryEncoder::Counter(int64_t counter)
    {
      lock_guard<mutex>  lock(_counterMutex);
      _counter = counter;
      if (_counter < _minObservedCounter) {
        _minObservedCounter = counter;
      }
      if (_counter > _maxObservedCounter) {
        _maxObservedCounter = counter;
      }
      return _counter;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    int64_t RotaryEncoder::MinObservedCounter() const
    {
      lock_guard<mutex>  lock(_counterMutex);
      return _minObservedCounter;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    int64_t RotaryEncoder::MaxObservedCounter() const
    {
      lock_guard<mutex>  lock(_counterMutex);
      return _maxObservedCounter;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    int64_t RotaryEncoder::Travel() const
    {
      lock_guard<mutex>  lock(_counterMutex);
      return (_maxObservedCounter - _minObservedCounter);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string RotaryEncoder::StateName(StateEnum stateValue)
    {
      static const map<StateEnum,string>  stateNames = {
        { e_stateInitial,                "Initial" },
        { e_stateMovingClockwise,        "MovingClockwise" },
        { e_stateMovingCounterClockwise, "MovingCounterClockwise" },
        { e_stateStopped,                "Stopped" }
      };
      string  rc;
      auto  sni = stateNames.find(stateValue);
      if (sni != stateNames.end()) {
        rc = sni->second;
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RotaryEncoder::StateName(StateFunc state) const
    {
      static const string stateNames[] = {
        "Initial",
        "MovingClockwise",
        "MovingCounterClockwise",
        "Stopped",
        "Unknown"
      };

      const string *rc = &(stateNames[4]);
      if (state == &RotaryEncoder::Stopped) {
        rc = &(stateNames[3]);
      }
      else if (state == &RotaryEncoder::MovingClockwise) {
        rc = &(stateNames[1]);
      }
      else if (state == &RotaryEncoder::MovingCounterClockwise) {
        rc = &(stateNames[2]);
      }
      else if (state == &RotaryEncoder::Initial) {
        rc = &(stateNames[0]);
      }

      return *rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::Initial(EncoderEventEnum event)
    {
      switch (event) {
        case e_movedClockwise:
          ChangeState(&RotaryEncoder::MovingClockwise);
          break;
        case e_movedCounterClockwise:
          ChangeState(&RotaryEncoder::MovingCounterClockwise);
          break;
        case e_stopped:
          ChangeState(&RotaryEncoder::Stopped);
          break;
        default:
          break;
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::MovingClockwise(EncoderEventEnum event)
    {
      switch (event) {
        case e_movedCounterClockwise:
          ChangeState(&RotaryEncoder::MovingCounterClockwise);
          break;
        case e_stopped:
          ChangeState(&RotaryEncoder::Stopped);
          break;
        default:
          break;
      }
      return;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::MovingCounterClockwise(EncoderEventEnum event)
    {
      switch (event) {
        case e_movedClockwise:
          ChangeState(&RotaryEncoder::MovingClockwise);
          break;
        case e_stopped:
          ChangeState(&RotaryEncoder::Stopped);
          break;
        default:
          break;
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::Stopped(EncoderEventEnum event)
    {
      switch (event) {
        case e_movedClockwise:
          ChangeState(&RotaryEncoder::MovingClockwise);
          break;
        case e_movedCounterClockwise:
          ChangeState(&RotaryEncoder::MovingCounterClockwise);
          break;
        default:
          break;
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RotaryEncoder::StateFunc RotaryEncoder::State()
    {
      lock_guard<mutex>  lock(_stateMutex);
      StateFunc  rc = _state;
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RotaryEncoder::Start(uint16_t readsPerSecond)
    {
      bool  rc = false;
      _reader.AddWatcher(this);
      _run = true;
      _thread = thread(&RotaryEncoder::Run, this);
      pthread_set_name_np(_thread.native_handle(), "encoder");
      if (_reader.Start(readsPerSecond)) {
        rc = true;
        Syslog(LOG_INFO, "RotaryEncoder(%s) started",
               _reader.DeviceName().c_str());
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::Stop()
    {
      _reader.Stop();
      _run = false;
      if (_thread.joinable()) {
        _thread.join();
        Syslog(LOG_INFO, "RotaryEncoder(%s) stopped",
               _reader.DeviceName().c_str());
      }
      _reader.RemoveWatcher(this);
      lock_guard<mutex>  lock(_stateMutex);
      _state = &RotaryEncoder::Initial;
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::ChangeState(RotaryEncoder::StateFunc state)
    {
      lock_guard<mutex>  lock(_stateMutex);
      if (_state != state) {
        if (SysLogger::MinimumPriority() >= LOG_DEBUG) {
          Syslog(LOG_DEBUG, "RotaryEncoder(%s) state %s -> %s",
                 _reader.DeviceName().c_str(), StateName(_state).c_str(),
                 StateName(state).c_str());
        }
        _state = state;
        EmitStateChange();
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::EmitStateChange() const
    {
      for (auto watcher : _watchers) {
        watcher->EncoderChangedState(_state);
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::Run()
    {
      while (_run) {
        EncoderEventEnum  event;
        if (_moves.TimedWaitForNotEmpty(chrono::milliseconds(500))) {
          bool  clockwise;
          if (_moves.PopFront(clockwise)) {
            if (clockwise) {
              (this->*_state)(e_movedClockwise);
              IncrementCounter();
            }
            else {
              (this->*_state)(e_movedCounterClockwise);
              DecrementCounter();
            }
          }
        }
        else {
          (this->*_state)(e_stopped);
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::IncrementCounter()
    {
      lock_guard<mutex>  lock(_counterMutex);
      ++_counter;
      if (_counter > _maxObservedCounter) {
        _maxObservedCounter = _counter;
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::DecrementCounter()
    {
      lock_guard<mutex>  lock(_counterMutex);
      --_counter;
      if (_counter < _minObservedCounter) {
        _minObservedCounter = _counter;
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::AddWatcher(RotaryEncoderWatcher *watcher)
    {
      _watchers.insert(watcher);
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void RotaryEncoder::RemoveWatcher(RotaryEncoderWatcher *watcher)
    {
      auto  it = _watchers.find(watcher);
      if (it != _watchers.end()) {
        _watchers.erase(watcher);
      }
      return;
    }

    
  }  // namespace Pi

}  // namespace Dwm
