//===========================================================================
// @(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/src/DwmPiPinReader.cc 8748 $
// @(#) $Id: DwmPiPinReader.cc 8748 2016-07-31 03:56:52Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPiPinReader.cc
//!  \brief Dwm::Pi::PinReader class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <unistd.h>
  #include <pthread_np.h>
}

#include <iostream>

#include "DwmPacer.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmPiPinReader.hh"

static const Dwm::SvnTag  svntag("@(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/src/DwmPiPinReader.cc 8748 $");

using namespace std;

namespace Dwm {

  namespace Pi {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    PinReader::PinReader(const GpioPin & pin, PinDebouncer<uint16_t> *debouncer)
        : _pin(pin), _debouncer(debouncer), _stateMutex(),
          _state(&PinReader::Initial), _watchersMutex(), _watchers(),
          _thread(), _run(false), _readsPerSecond(20)
    {
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    PinReader::~PinReader()
    {
      Stop();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    PinReader::StateFunc PinReader::State() const
    {
      lock_guard<mutex>  lock(_stateMutex);
      StateFunc  rc = _state;
      return rc;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & PinReader::StateName(StateFunc state) const
    {
      static const string stateNames[] = {
        "Initial",
        "High",
        "Low",
        "Unknown"
      };
      if (state == &PinReader::Initial) {
        return stateNames[0];
      }
      else if (state == &PinReader::High) {
        return stateNames[1];
      }
      else if (state == &PinReader::Low) {
        return stateNames[2];
      }
      else {
        return stateNames[3];
      }
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void PinReader::Initial(uint8_t val)
    {
      switch (val) {
        case 0:
          ChangeState(&PinReader::Low);
          break;
        case 1:
          ChangeState(&PinReader::High);
          break;
        default:
          break;
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void PinReader::High(uint8_t val)
    {
      if (! val) {
        ChangeState(&PinReader::Low);
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void PinReader::Low(uint8_t val)
    {
      if (val) {
        ChangeState(&PinReader::High);
      }
      return;
    }
        
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void PinReader::AddWatcher(PinWatcher *watcher)
    {
      lock_guard<mutex>  lock(_watchersMutex);
      _watchers.insert(watcher);
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void PinReader::RemoveWatcher(PinWatcher *watcher)
    {
      lock_guard<mutex>  lock(_watchersMutex);
      auto  it = _watchers.find(watcher);
      if (it != _watchers.end()) {
        _watchers.erase(it);
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PinReader::Start(uint16_t readsPerSecond)
    {
      bool  rc = false;
      _readsPerSecond = readsPerSecond;
      //  Start the pin polling thread
      _run = true;
      _thread = thread(&PinReader::Run, this);
      pthread_set_name_np(_thread.native_handle(), "pin reader");
      rc = true;
      Syslog(LOG_INFO, "PinReader(%s) started", _pin.Name().c_str());
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void PinReader::Stop()
    {
      _run = false;
      if (_thread.joinable()) {
        _thread.join();
        lock_guard<mutex>  lock(_stateMutex);
        _state = &PinReader::Initial;
        Syslog(LOG_INFO, "PinReader(%s) stopped", _pin.Name().c_str());
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void
    PinReader::ChangeState(PinReader::StateFunc state)
    {
      lock_guard<mutex>  lock(_stateMutex);
      if (_state != state) {
        if (SysLogger::MinimumPriority() >= LOG_DEBUG) {
          Syslog(LOG_DEBUG, "PinReader(%s) state %s -> %s",
                 _pin.Name().c_str(), StateName(_state).c_str(),
                 StateName(state).c_str());
        }
        _state = state;
        if (state == &PinReader::High) {
          EmitHighEvent();
        }
        else {
          EmitLowEvent();
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void PinReader::EmitHighEvent()
    {
      lock_guard<mutex>   lock(_watchersMutex);
      if (! _watchers.empty()) {
        for (auto watcher : _watchers) {
          watcher->PinWentHigh();
        }
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void PinReader::EmitLowEvent()
    {
      lock_guard<mutex>   lock(_watchersMutex);
      if (! _watchers.empty()) {
        for (auto watcher : _watchers) {
          watcher->PinWentLow();
        }
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void PinReader::Run()
    {
      Pacer    pacer(_readsPerSecond);
      uint8_t  val;
      uint8_t  prevVal = 0xFF;
      while (_run) {
        val = _pin.Value() ? 1 : 0;
        if (_debouncer != nullptr) {
          val = _debouncer->Debounce(val ? 255 : 0);
        }
        if (val != prevVal) {
          (this->*_state)(val);
          prevVal = val;
        }
        if (pacer.Rate()) {
          pacer.Pace();
        }
      }
      return;
    }
    
  }  // namespace Pi
  
}  // namespace Dwm
