//===========================================================================
// @(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/src/DwmPiGpio.cc 8526 $
// @(#) $Id: DwmPiGpio.cc 8526 2016-05-13 01:58:41Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPiGpio.cc
//!  \brief Dwm::Pi::GpioDevice and Dwm::Pi::GpioPin implementations
//---------------------------------------------------------------------------

#include <cassert>
#include <iostream>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmPiGpio.hh"

static const Dwm::SvnTag  svntag("@(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/src/DwmPiGpio.cc 8526 $");

using namespace std;

namespace Dwm {

  namespace Pi {

    std::map<gpio_handle_t,std::mutex>  GpioPin::_deviceMutexes;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    GpioDevice::GpioDevice(unsigned int unit)
    {
      _handle = gpio_open(unit);
      assert (_handle >= 0);
      GpioPin::_deviceMutexes[_handle];
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    GpioDevice::GpioDevice(const std::string & device)
    {
      _handle = gpio_open_device(device.c_str());
      assert(_handle >= 0);
      GpioPin::_deviceMutexes[_handle];
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    GpioDevice::~GpioDevice()
    {
      if (_handle >= 0) {
        gpio_close(_handle);
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    gpio_handle_t & GpioDevice::Handle()
    {
      return _handle;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    GpioPin::GpioPin(GpioDevice & device, gpio_pin_t pin, bool input)
        : _device(device), _pin(pin)
    {
      if (input) {
        Direction(e_directionInput);
      }
      else {
        Direction(e_directionOutput);
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string GpioPin::Name() const
    {
      string           rc;
      struct gpio_pin  gpioPin;
      gpioPin.gp_pin = _pin;
      lock_guard<mutex>   lock(_deviceMutexes[_device.Handle()]);
      if (ioctl(_device.Handle(), GPIOGETCONFIG, &gpioPin) == 0) {
        rc = gpioPin.gp_name;
      }
      else {
        Syslog(LOG_ERR, "ioctl(%d,GPIOGETCONFIG,%p) failed for pin %d",
               _device.Handle(), &gpioPin, _pin);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::Name(const std::string & name)
    {
      lock_guard<mutex>   lock(_deviceMutexes[_device.Handle()]);
      return (gpio_pin_set_name(_device.Handle(), _pin, (char *)name.c_str())
              == 0);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    GpioPin::DirectionEnum GpioPin::Direction()
    {
      DirectionEnum  rc = e_directionUnknown;
      uint32_t       flags;
      if (GetFlags(flags)) {
        if (flags & GPIO_PIN_INPUT) {
          rc = e_directionInput;
        }
        else if (flags & GPIO_PIN_OUTPUT) {
          rc = e_directionOutput;
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::Direction(DirectionEnum direction)
    {
      bool  rc = false;
      if (direction == e_directionOutput) {
        lock_guard<mutex>   lock(_deviceMutexes[_device.Handle()]);
        rc = (gpio_pin_output(_device.Handle(), _pin) == 0 ? true : false);
      }
      else {
        lock_guard<mutex>   lock(_deviceMutexes[_device.Handle()]);
        rc = (gpio_pin_input(_device.Handle(), _pin) == 0 ? true : false);
      }
      if (! rc) {
        Syslog(LOG_ERR, "SetDirection() failed for pin %d", _pin);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::Value() const
    {
      lock_guard<mutex>   lock(_deviceMutexes[_device.Handle()]);
      return (gpio_pin_get(_device.Handle(), _pin) == 1 ? true : false);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::Value(bool high)
    {
      bool  rc = false;
      lock_guard<mutex>   lock(_deviceMutexes[_device.Handle()]);
      if (gpio_pin_set(_device.Handle(), _pin, (high ? 1 : 0)) == 0) {
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::Toggle()
    {
      lock_guard<mutex>   lock(_deviceMutexes[_device.Handle()]);
      return (gpio_pin_toggle(_device.Handle(), _pin) == 0);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::Pull(PullEnum pull)
    {
      bool  rc = false;
      gpio_config_t  pinCfg;
      bzero(&pinCfg, sizeof(pinCfg));
      pinCfg.g_pin = _pin;
      lock_guard<mutex>   lock(_deviceMutexes[_device.Handle()]);
      if (gpio_pin_config(_device.Handle(), &pinCfg) == 0) {
        if (pinCfg.g_flags & GPIO_PIN_INPUT) {
          switch (pull) {
            case e_pullNone:
              pinCfg.g_flags &= ~(GPIO_PIN_PULLUP|GPIO_PIN_PULLDOWN);
              break;
            case e_pullUp:
              pinCfg.g_flags &= ~(GPIO_PIN_PULLUP|GPIO_PIN_PULLDOWN);
              pinCfg.g_flags |= GPIO_PIN_PULLUP;
              break;
            case e_pullDown:
              pinCfg.g_flags &= ~(GPIO_PIN_PULLUP|GPIO_PIN_PULLDOWN);
              pinCfg.g_flags |= GPIO_PIN_PULLDOWN;
              break;
            default:
              assert("Invalid PullEnum" == 0);
              break;
          }
          if (gpio_pin_set_flags(_device.Handle(), &pinCfg) == 0) {
            rc = true;
          }
          else {
            Syslog(LOG_ERR, "gpio_pin_set_flags() failed for pin %d", _pin);
          }
        }
      }
      else {
        Syslog(LOG_ERR, "gpio_pin_config() failed for pin %d", _pin);
      }
      
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    GpioPin::PullEnum GpioPin::Pull() const
    {
      PullEnum  rc = e_pullNone;
      uint32_t  flags;
      if (GetFlags(flags)) {
        if (flags & GPIO_PIN_INPUT) {
          if (flags & GPIO_PIN_PULLUP) {
            rc = e_pullUp;
          }
          if (flags & GPIO_PIN_PULLDOWN) {
            rc = e_pullDown;
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::InvertInput() const
    {
      bool      rc = false;
      uint32_t  flags;
      if (GetFlags(flags) && (flags & GPIO_PIN_INVIN)) {
        rc = true;
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::InvertInput(bool invert)
    {
      bool  rc = false;
      if (invert) {
        rc = SetFlags(GPIO_PIN_INVIN);
      }
      else {
        rc = ClearFlags(GPIO_PIN_INVIN);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::InvertOutput() const
    {
      bool      rc = false;
      uint32_t  flags;
      if (GetFlags(flags) && (flags & GPIO_PIN_INVOUT)) {
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::InvertOutput(bool invert)
    {
      bool  rc = false;
      if (invert) {
        rc = SetFlags(GPIO_PIN_INVOUT);
      }
      else {
        rc = ClearFlags(GPIO_PIN_INVOUT);
      }
      return rc;
    }
    

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::GetFlags(uint32_t & flags) const
    {
      bool  rc = false;
      gpio_config_t  pinCfg;
      bzero(&pinCfg, sizeof(pinCfg));
      pinCfg.g_pin = _pin;
      lock_guard<mutex>   lock(_deviceMutexes[_device.Handle()]);
      if (gpio_pin_config(_device.Handle(), &pinCfg) == 0) {
        flags = pinCfg.g_flags;
        rc = true;
      }
      else {
        Syslog(LOG_ERR, "gpio_pin_config() failed for pin %d", _pin);
      }
      
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::SetFlags(uint32_t flags)
    {
      bool  rc = false;
      gpio_config_t  pinCfg;
      bzero(&pinCfg, sizeof(pinCfg));
      pinCfg.g_pin = _pin;
      lock_guard<mutex>   lock(_deviceMutexes[_device.Handle()]);
      if (gpio_pin_config(_device.Handle(), &pinCfg) == 0) {
        pinCfg.g_flags |= flags;
        if (gpio_pin_set_flags(_device.Handle(), &pinCfg) == 0) {
          rc = true;
        }
        else {
          Syslog(LOG_ERR, "gpio_pin_set_flags() failed for pin %d", _pin);
        }
      }
      else {
        Syslog(LOG_ERR, "gpio_pin_config() failed for pin %d", _pin);
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool GpioPin::ClearFlags(uint32_t flags)
    {
      bool  rc = false;
      gpio_config_t  pinCfg;
      bzero(&pinCfg, sizeof(pinCfg));
      pinCfg.g_pin = _pin;
      lock_guard<mutex>   lock(_deviceMutexes[_device.Handle()]);
      if (gpio_pin_config(_device.Handle(), &pinCfg) == 0) {
        pinCfg.g_flags &= (~flags);
        if (gpio_pin_set_flags(_device.Handle(), &pinCfg) == 0) {
          rc = true;
        }
        else {
          Syslog(LOG_ERR, "gpio_pin_set_flags() failed for pin %d", _pin);
        }
      }
      else {
        Syslog(LOG_ERR, "gpio_pin_config() failed for pin %d", _pin);
      }
      return rc;
    }

    
  }  // namespace Pi

}  // namespace Dwm
