//===========================================================================
// @(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/include/DwmPiRotaryEncoderReader.hh 9711 $
// @(#) $Id: DwmPiRotaryEncoderReader.hh 9711 2017-07-04 06:05:54Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPiRotaryEncoderReader.hh
//!  \brief RotaryEncoderReader class definition
//!  \author Daniel W. McRobb \link mailto:dwm@mcplex.net \endlink
//---------------------------------------------------------------------------

#ifndef _DWMPIROTARYENCODERREADER_HH_
#define _DWMPIROTARYENCODERREADER_HH_

extern "C" {
  #include <sys/types.h>
  #include <libgpio.h>
}

#include <map>
#include <thread>
#include <set>

#include "DwmPiGpio.hh"

namespace Dwm {

  namespace Pi {

    //------------------------------------------------------------------------
    //!  Interface for class that's interested in receiving rotary encoder
    //!  events.
    //------------------------------------------------------------------------
    class RotaryEncoderReaderWatcher
    {
    public:
      //----------------------------------------------------------------------
      //!  Called when the associated RotaryEncodeReader moves clockwise.
      //----------------------------------------------------------------------
      virtual void EncoderMovedClockwise() = 0;
      
      //----------------------------------------------------------------------
      //!  Called when the associated RotaryEncoderReader moves
      //!  counter-clockwise.
      //----------------------------------------------------------------------
      virtual void EncoderMovedCounterClockwise() = 0;
    };
      
    //------------------------------------------------------------------------
    //!  Encapsulates a state machine and thread for a rotary encoder reader.
    //!  An instance of the class may be used to watch for rotary encoder
    //!  state changes.  As a state machine, it only changes state when a
    //!  valid position change occurs.  This helps avoid propagating noise
    //!  to watchers of an instance of this class.
    //!
    //!  When the state of a RotaryEncoderReader changes, it will call
    //!  the EncoderMovedClockwise() or EncoderMovedCounterClockwise()
    //!  members of any objects registered with AddWatcher().  These 'watcher'
    //!  objects must have inherited from RotaryEncoderReaderWatcher, which
    //!  is a pure virtual class.
    //!
    //!  A diagram of the RotaryEncoderReader state machine is shown below.
    //!
    //!  \dotfile RotaryEncoderReaderStateMachine.dot
    //------------------------------------------------------------------------
    class RotaryEncoderReader
    {
    public:
      //----------------------------------------------------------------------
      //!  Type definition for a state of the RotaryEncoderReader.  This
      //!  is a pointer to member function which takes a combintation of
      //!  the A and B channels of the encoder as an argument ((A << 1) | B).
      //----------------------------------------------------------------------
      typedef void (Dwm::Pi::RotaryEncoderReader::*StateFunc)(uint8_t ab);

      //----------------------------------------------------------------------
      //!  Construct from device name (e.g. "gpiorotenc0").
      //----------------------------------------------------------------------
      RotaryEncoderReader(const std::string & device);

      //----------------------------------------------------------------------
      //!  Destructor.
      //----------------------------------------------------------------------
      ~RotaryEncoderReader();

      //----------------------------------------------------------------------
      //!  Returns the rotary encoder device name.
      //----------------------------------------------------------------------
      const std::string & DeviceName() const;
      
      //----------------------------------------------------------------------
      //!  Returns a string representation of the given @c state of a
      //!  RotaryEncoderRaader.
      //----------------------------------------------------------------------
      const std::string & StateName(StateFunc state) const;

      //----------------------------------------------------------------------
      //!  The Initial state of the RotaryEncoderReader.
      //----------------------------------------------------------------------
      void Initial(uint8_t ab);
      
      //----------------------------------------------------------------------
      //!  The 00 (both channels low) state of the RotaryEncoderReader.
      //----------------------------------------------------------------------
      void State00(uint8_t ab);
      
      //----------------------------------------------------------------------
      //!  The 10 state (A high, B low) of the RotaryEncoderReader.
      //----------------------------------------------------------------------
      void State10(uint8_t ab);

      //----------------------------------------------------------------------
      //!  The 11 state (both channels high) of the RotaryEncoderReader.
      //----------------------------------------------------------------------
      void State11(uint8_t ab);
      
      //----------------------------------------------------------------------
      //!  The 01 state (A low, B high) of the RotaryEncoderReader.
      //----------------------------------------------------------------------
      void State01(uint8_t ab);

      //----------------------------------------------------------------------
      //!  Adds the given @c watcher to the set of watchers to be notified
      //!  when the RotaryEncoderReader changes state.
      //----------------------------------------------------------------------
      void AddWatcher(RotaryEncoderReaderWatcher *watcher);
      
      //----------------------------------------------------------------------
      //!  Removes the given @c watcher from the set of watchers to be
      //!  notified when the RotaryEncoderReader changes state.
      //----------------------------------------------------------------------
      void RemoveWatcher(RotaryEncoderReaderWatcher *receiver);
      
      //----------------------------------------------------------------------
      //!  Starts the RotaryEncoderReader.  This will start the thread for
      //!  the RotaryEncoderReader.
      //----------------------------------------------------------------------
      bool Start(uint16_t readsPerSecond = 2000);
      
      //----------------------------------------------------------------------
      //!  Stops the RotaryEncoderReader.  This will stop the thread for
      //!  the RotaryEncoderReader.
      //----------------------------------------------------------------------
      void Stop();
      
    private:
      std::string                             _device;
      int                                     _devfd;
      std::mutex                              _stateMutex;
      StateFunc                               _currentState;
      std::mutex                              _watchersMutex;
      std::set<RotaryEncoderReaderWatcher *>  _watchers;
      std::thread                             _thread;
      bool                                    _run;
      uint16_t                                _readsPerSecond;
      
      void ChangeState(RotaryEncoderReader::StateFunc state, bool clockwise);
      void EmitClockwiseEvent();
      void EmitCounterClockwiseEvent();
      void Run();
    };
    
  }  // namespace Pi

}  // namespace Dwm

#endif  // _DWMPIROTARYENCODERREADER_HH_
