//===========================================================================
// @(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/include/DwmPiRotaryEncoder.hh 9712 $
// @(#) $Id: DwmPiRotaryEncoder.hh 9712 2017-07-04 06:08:17Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPiRotaryEncoder.hh
//!  \brief RotaryEncoder class definition
//!  \author Daniel W. McRobb <dwm@mcplex.net>
//---------------------------------------------------------------------------

#ifndef _DWMPIROTARYENCODER_HH_
#define _DWMPIROTARYENCODER_HH_

#include <set>

#include "DwmThreadQueue.hh"
#include "DwmPiRotaryEncoderReader.hh"

namespace Dwm {

  namespace Pi {

    class RotaryEncoderWatcher;

    //------------------------------------------------------------------------
    //!  Encapsulates a rotary encoder.  This is simply a state machine
    //!  with states Initial, MovingClockwise, MovingCounterClockwise and
    //!  Stopped.  It contains a RotaryEncoderReader for reading the
    //!  encoder channels A and B.  Once started, the RotaryEncoder will
    //!  run a thread to monitor events from the RotaryEncoderReader and
    //!  change states appropriately.
    //!  \dotfile RotaryEncoderStateMachine.dot
    //------------------------------------------------------------------------
    class RotaryEncoder
      : public RotaryEncoderReaderWatcher
    {
    public:
      //----------------------------------------------------------------------
      //!  Inherited from RotaryEncoderWatcher.  Processes clockwise
      //!  move events from the RotaryEncoderReader.
      //----------------------------------------------------------------------
      void EncoderMovedClockwise();

      //----------------------------------------------------------------------
      //!  Inherited from RotaryEncoderWatcher.  Processes counterclockwise
      //!  move events from the RotaryEncoderReader.
      //----------------------------------------------------------------------
      void EncoderMovedCounterClockwise();
      
      //----------------------------------------------------------------------
      //!  Events propagated to the state machine from the
      //!  contained RotaryEncoderReader.
      //----------------------------------------------------------------------
      typedef enum {
        e_movedClockwise,
        e_movedCounterClockwise,
        e_stopped
      } EncoderEventEnum;

      //----------------------------------------------------------------------
      //!  State of the encoder as an enumerated value.
      //----------------------------------------------------------------------
      typedef enum {
        e_stateInitial,
        e_stateMovingClockwise,
        e_stateMovingCounterClockwise,
        e_stateStopped
      } StateEnum;
        
      //----------------------------------------------------------------------
      //!  Type definition of our state (member function pointer).
      //----------------------------------------------------------------------
      typedef void (Dwm::Pi::RotaryEncoder::*StateFunc)(EncoderEventEnum);

      //----------------------------------------------------------------------
      //!  Construct with the given @c device.
      //----------------------------------------------------------------------
      RotaryEncoder(const std::string & device);

      //----------------------------------------------------------------------
      //!  Destructor.  Stops the encoder.
      //----------------------------------------------------------------------
      ~RotaryEncoder();

      //----------------------------------------------------------------------
      //!  Returns a const reference to the encapsulated RotaryEncoderReader.
      //----------------------------------------------------------------------
      const RotaryEncoderReader & Reader() const;
      
      //----------------------------------------------------------------------
      //!  Returns the state of the encoder as a StateEnum.
      //----------------------------------------------------------------------
      StateEnum StateValue() const;

      //----------------------------------------------------------------------
      //!  Returns the state name associated with the given @c stateValue.
      //----------------------------------------------------------------------
      static std::string StateName(StateEnum stateValue);
      
      //----------------------------------------------------------------------
      //!  Returns a string representation of the given @c state.
      //----------------------------------------------------------------------
      const std::string & StateName(StateFunc state) const;

      //----------------------------------------------------------------------
      //!  Returns the counter.
      //----------------------------------------------------------------------
      int64_t Counter() const;

      //----------------------------------------------------------------------
      //!  Sets the counter.
      //----------------------------------------------------------------------
      int64_t Counter(int64_t counter);

      //----------------------------------------------------------------------
      //!  Returns the minimum observed counter value.
      //----------------------------------------------------------------------
      int64_t MinObservedCounter() const;

      //----------------------------------------------------------------------
      //!  Returns the maximum observed counter value.
      //----------------------------------------------------------------------
      int64_t MaxObservedCounter() const;

      //----------------------------------------------------------------------
      //!  Returns the distance between the maximum and minimum observed
      //!  counters.  Should always be greater than or equal to 0.
      //----------------------------------------------------------------------
      int64_t Travel() const;
      
      //----------------------------------------------------------------------
      //!  The initial state of the encoder.
      //----------------------------------------------------------------------
      void Initial(EncoderEventEnum event);
      
      //----------------------------------------------------------------------
      //!  The moving clockwise state fo the encoder.
      //----------------------------------------------------------------------
      void MovingClockwise(EncoderEventEnum event);
      
      //----------------------------------------------------------------------
      //!  The moving counterclockwise state of the encoder.
      //----------------------------------------------------------------------
      void MovingCounterClockwise(EncoderEventEnum event);

      //----------------------------------------------------------------------
      //!  The stopped state of the encoder.
      //----------------------------------------------------------------------
      void Stopped(EncoderEventEnum event);

      //----------------------------------------------------------------------
      //!  Returns the current state of the encoder.
      //----------------------------------------------------------------------
      StateFunc State();

      //----------------------------------------------------------------------
      //!  Starts the encoder, which also starts the underlying
      //!  RotaryEncoderReader.
      //----------------------------------------------------------------------
      bool Start(uint16_t readsPerSecond = 2000);

      //----------------------------------------------------------------------
      //!  Stops the encoder, which also stops the underlying
      //!  RotaryEncoderReader.
      //----------------------------------------------------------------------
      void Stop();

      //----------------------------------------------------------------------
      //!  Adds the given @c watcher to the set of watchers that will be
      //!  informed of rotary encoder state changes.
      //----------------------------------------------------------------------
      void AddWatcher(RotaryEncoderWatcher *watcher);
      
      //----------------------------------------------------------------------
      //!  Removes the given @c watcher from the set of watchers that will be
      //!  informed of rotary encoder state changes.
      //----------------------------------------------------------------------
      void RemoveWatcher(RotaryEncoderWatcher *watcher);
      
    private:
      RotaryEncoderReader               _reader;
      std::mutex                        _stateMutex;
      StateFunc                         _state;
      Thread::Queue<bool>               _moves;
      mutable std::mutex                _counterMutex;
      int64_t                           _counter;
      int64_t                           _minObservedCounter;
      int64_t                           _maxObservedCounter;
      std::thread                       _thread;
      bool                              _run;
      std::set<RotaryEncoderWatcher *>  _watchers;
      
      void ChangeState(RotaryEncoder::StateFunc state);
      void EmitStateChange() const;
      void Run();
      void IncrementCounter();
      void DecrementCounter();
    };

    //------------------------------------------------------------------------
    //!  Interface for a class that can watch for rotary encoder state
    //!  changes.
    //------------------------------------------------------------------------
    class RotaryEncoderWatcher
    {
    public:
      //----------------------------------------------------------------------
      //!  If the watcher is added to a RotaryEncoder via AddWatcher(),
      //!  this member will be called whenever the RotaryEncoder changes
      //!  state.  Note that this call is made from the context of the
      //!  RotaryEncoder's thread, so take care to make this member
      //!  threadsafe in your derived class.
      //----------------------------------------------------------------------
      virtual void EncoderChangedState(RotaryEncoder::StateFunc state) = 0;
    };
    

      
  }  // namespace Pi

}  // namespace Dwm

#endif  // _DWMPIROTARYENCODER_HH_
