//===========================================================================
// @(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/include/DwmPiPinReader.hh 9712 $
// @(#) $Id: DwmPiPinReader.hh 9712 2017-07-04 06:08:17Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPiPinReader.hh
//!  \brief PinReader class definition
//!  \author Daniel W. McRobb <dwm@mcplex.net>
//---------------------------------------------------------------------------

#ifndef _DWMPIPINREADER_HH_
#define _DWMPIPINREADER_HH_

extern "C" {
  #include <sys/types.h>
  #include <libgpio.h>
}

#include <map>
#include <thread>
#include <set>

#include "DwmPiGpio.hh"
#include "DwmPiPinDebouncer.hh"

namespace Dwm {

  namespace Pi {

    //------------------------------------------------------------------------
    //!  Interface for class that's interested in receiving pin change
    //!  events.
    //------------------------------------------------------------------------
    class PinWatcher
    {
    public:
      virtual void PinWentLow() = 0;
      virtual void PinWentHigh() = 0;
    };
      
    //------------------------------------------------------------------------
    //!  Encapsulates a pin reader.  This class allows monitoring a pin for
    //!  changes in state (high or low).  It runs a thread to do this, and
    //!  it should be noted that calls to watchers' PinWentLow() and
    //!  PinWentHigh() happen in thread context.  Hence a watcher should
    //!  take care to make sure that their PinWentLow() and PinWentHigh()
    //!  methods are threadsafe.
    //!  \dotfile PinReaderStateMachine.dot "Pin reader state machine"
    //------------------------------------------------------------------------
    class PinReader
    {
    public:
      //----------------------------------------------------------------------
      //!  Typedef for the pin reader state (member function pointer).
      //----------------------------------------------------------------------
      typedef void (Dwm::Pi::PinReader::*StateFunc)(uint8_t ab);

      //----------------------------------------------------------------------
      //!  Construct from a GpioPin.  Note that @c pin can be an input or
      //!  output pin, but it should be configured before calling this
      //!  constructor, and the GpioDevice should be open as well.  The
      //!  PinReader class makes no changes to the configuration of the given
      //!  @c pin.
      //----------------------------------------------------------------------
      PinReader(const GpioPin & pin,
                PinDebouncer<uint16_t> *debouncer = nullptr);

      //----------------------------------------------------------------------
      //!  Destructor.  Will stop the pin reader's thread.
      //----------------------------------------------------------------------
      ~PinReader();
      
      //----------------------------------------------------------------------
      //!  Returns the pin that the pin reader is monitoring.
      //----------------------------------------------------------------------
      const GpioPin & Pin() const
      {
        return _pin;
      }
      
      //----------------------------------------------------------------------
      //!  The initial state of the pin reader (before the pin has been
      //!  read).
      //----------------------------------------------------------------------
      void Initial(uint8_t val);
      
      //----------------------------------------------------------------------
      //!  The high state of the pin reader.  In this state, the last pin
      //!  reading was high (1).
      //----------------------------------------------------------------------
      void High(uint8_t val);
      
      //----------------------------------------------------------------------
      //!  the low state of the pin reader.  In this state, the last pin
      //!  reading was low (0).
      //----------------------------------------------------------------------
      void Low(uint8_t val);

      //----------------------------------------------------------------------
      //!  Returns the current state of the pin (according to the last pin
      //!  reading).
      //----------------------------------------------------------------------
      StateFunc State() const;
      
      //----------------------------------------------------------------------
      //!  Returns a string representation of the given pin reader @c state.
      //----------------------------------------------------------------------
      const std::string & StateName(StateFunc state) const;

      //----------------------------------------------------------------------
      //!  Adds a watcher to the set of watchers to be notified when the
      //!  pin changes state.
      //----------------------------------------------------------------------
      void AddWatcher(PinWatcher *watcher);
      
      //----------------------------------------------------------------------
      //!  Removes a watcher from the set of watchers to be notified when
      //!  the pin changes state.
      //----------------------------------------------------------------------
      void RemoveWatcher(PinWatcher *watcher);
      
      //----------------------------------------------------------------------
      //!  Starts the pin reader (activates its thread that reads the pin).
      //!  The pin will be read at the given rate of @c readsPerSecond.
      //----------------------------------------------------------------------
      bool Start(uint16_t readsPerSecond = 20);
      
      //----------------------------------------------------------------------
      //!  Stops the pin reader (stops its thread that reads the pin).
      //----------------------------------------------------------------------
      void Stop();
      
    private:
      GpioPin                  _pin;
      PinDebouncer<uint16_t>  *_debouncer;
      mutable std::mutex       _stateMutex;
      StateFunc                _state;
      std::mutex               _watchersMutex;
      std::set<PinWatcher *>   _watchers;
      std::thread              _thread;
      bool                     _run;
      uint16_t                 _readsPerSecond;
      
      void ChangeState(StateFunc state);
      void EmitLowEvent();
      void EmitHighEvent();
      void Run();
    };
    
  }  // namespace Pi

}  // namespace Dwm

#endif  // _DWMPIPINREADER_HH_
