//===========================================================================
// @(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/include/DwmPiPinDebouncer.hh 8748 $
// @(#) $Id: DwmPiPinDebouncer.hh 8748 2016-07-31 03:56:52Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2009, 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPiPinDebouncer.hh
//!  \brief Dwm::Pi::PinDebouncer class definition
//---------------------------------------------------------------------------

#ifndef _DWMPIPINDEBOUNCER_HH_
#define _DWMPIPINDEBOUNCER_HH_

#include <cstdint>

#include "DwmPiLowPassFilter.hh"

namespace Dwm {

  namespace Pi {

    //------------------------------------------------------------------------
    //!  This is a first-order low-pass filter (exponentially weighted moving
    //!  average filter) in series with a software Schmitt trigger.  It is
    //!  intended for debouncing digital inputs.
    //!
    //!  The weighted moving average filter is the usual:
    //!
    //!     x[k] = filter_constant * x[k-1] + (1 - filter_constant) * x[k]
    //!
    //!  The filter constant is .75 by default (@c FCDividend = 3,
    //!  @c FCDivisor = LowPassFilterDivisor4), but done using fixed-point
    //!  arithmetic.  See the Dwm::Avr::LowPassFilter documentation for more
    //!  information on the low-pass filter.
    //!
    //!  Hardware Schmitt triggers typically use thresholds around 1/3 and
    //!  2/3 of Vcc.  Here we're using @c LowerThreshold and
    //!  @c UpperThreshold.  Some examples of the number of consecutive
    //!  values of 0 or 1 that must occur on the input @c in to cause a
    //!  transition on the output @ cout for different values of
    //!  @c LowerThreshold and @c UpperThreshold with @c FCDividend = 3 and
    //!  @c FCDivisor = LowPassFilterDivisor4:
    //!
    //!  LowerThreshold             UpperThreshold                  count
    //!  -------------------------  -----------------------------  --------
    //!  120                        135                             3 or 4
    //!  85  (85/255 = .33333)      170 (170/255 = .66667)          4 or 5
    //!  64  (64/255 = .25)         192 (192/255 = .75)               6
    //!  51  (51/255 = .2)          204 (204/255 = .8)                7
    //!  30                         225                               9
    //!  15                         240                            11 or 12
    //!
    //!  There is no rule that you must use thresholds centered around 127.
    //!  Centering elsewhere will cause one output result to be easier to reach
    //!  than the other.
    //------------------------------------------------------------------------
    template <typename InputType = uint16_t>
    class PinDebouncer
    {
    public:
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      PinDebouncer(InputType initialValue, uint8_t fcDividend = 3,
                   LowPassFilterDivisor fcDivisor = LowPassFilterDivisor4,
                   InputType lowerThresh = 15, InputType upperThresh = 240)
          : _lowPassFilter(lowerThresh + ((upperThresh - lowerThresh)/2),
                           fcDividend, fcDivisor),
            _lowerThresh(lowerThresh), _upperThresh(upperThresh),
            _value(initialValue)
      { }
    
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      uint8_t Debounce(InputType in)
      {
        InputType  filteredValue = _lowPassFilter.Filter(in);
        if (filteredValue < _lowerThresh) {
          _value = 0;
        }
        else if (filteredValue > _upperThresh) {
          _value = 1;
        }
        return(_value);
      }

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      uint8_t Value() const
      {
        return(_value);
      }
      
    private:
      LowPassFilter<InputType>  _lowPassFilter;
      InputType                 _lowerThresh;
      InputType                 _upperThresh;
      uint8_t                   _value;
    };

    //------------------------------------------------------------------------
    //!  A simple debouncer with a fixed low-pass filter constant of .75, a 
    //!  fixed lower trigger threshold of 15 and a fixed upper trigger 
    //!  threshold of 240.
    //!  An application can repeatedly call Debounce() to debounce a high/low
    //!  signal.  Debounce() and Value() will return 1 if the debounced
    //!  signal is high and 0 if the debounced signal is low.  This debouncer
    //!  is typically used for pin input where the pin is connected to 
    //!  a momentary switch or button.
    //------------------------------------------------------------------------
    class SimplePinDebouncer
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct with @c initialValue (0 or 1).
      //----------------------------------------------------------------------
      SimplePinDebouncer(uint8_t initialValue);
      
      //----------------------------------------------------------------------
      //!  Debounce the given input @c in (0 or non-zero) and return the
      //!  debounced result (0 or 1).
      //----------------------------------------------------------------------
      uint8_t Debounce(uint8_t in);

      //----------------------------------------------------------------------
      //!  Returns the current value (0 or 1).
      //----------------------------------------------------------------------
      uint8_t Value() const;
      
    private:
      uint8_t  _filterValue;
      uint8_t  _value;
    
      static const uint8_t  k_lowerThreshold = 15;
      static const uint8_t  k_upperThreshold = 240;
      SimplePinDebouncer();
    };
    
  }  // namespace Pi

}  // namespace Dwm

#endif  // _DWMPIPINDEBOUNCER_HH_
