//===========================================================================
// @(#) $DwmPath: dwm/libDwmPi/tags/libDwmPi-0.1.0/include/DwmPiLowPassFilter.hh 8748 $
// @(#) $Id: DwmPiLowPassFilter.hh 8748 2016-07-31 03:56:52Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2009, 2011, 2013, 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPiLowPassFilter.hh
//!  \brief Dwm::Pi::LowPassFilter class definition
//---------------------------------------------------------------------------

#ifndef _DWMPILOWPASSFILTER_HH_
#define _DWMPILOWPASSFILTER_HH_

#include <cstdint>

namespace Dwm {

  namespace Pi {

    //------------------------------------------------------------------------
    //!  Filter constant divisors.
    //------------------------------------------------------------------------
    typedef enum {
        LowPassFilterDivisor2  = 1,
        LowPassFilterDivisor4  = 2,
        LowPassFilterDivisor8  = 3,
        LowPassFilterDivisor16 = 4,
        LowPassFilterDivisor32 = 5,
        LowPassFilterDivisor64 = 6
      } LowPassFilterDivisor;
        
    //------------------------------------------------------------------------
    //!  Implements a typical exponentially weighted moving average filter,
    //!  i.e. a discrete first-order low-pass filter:
    //!
    //!    x[k] = (FC * x[k - 1]) + ((1 - FC) * x[k])
    //!
    //!  where FC is the filter constant.
    //!
    //!  The filter constant is a fraction with @c Dividend as the dividend
    //!  (numerator) and @c Divisor as the divisor (denominator).  I've
    //!  constrained the divisors (denominators) to powers of 2 for
    //!  efficiency (can use a right shift instead of a divide, saving
    //!  precious clock cycles).  @c Divisor is actually the number
    //!  of bits to shift to the right, not the actual divisor.  For example,
    //!  shifting the right by 4 is equivalent to dividing by 16.  Hence
    //!  LowPassFilterDivisor16 = 4.
    //!
    //!  This filter is mainly targetted at 10-bit ADC values.  In this case,
    //!  InputType would be uint16_t, and 16-bit math would be used.  This
    //!  leads to the constraint:
    //!  
    //!    ((prev * Dividend) + (in << Divisor)) must not exceed 0xFFFF.
    //!  
    //!  With a 10-bit ADC, the highest usable @c Dividend would be 31.
    //!  With a 12-bit ADC, the highest usable @c Dividend would be 7.
    //!
    //!  If InputType is uint32_t, the constraint becomes:
    //!
    //!    ((prev * Dividend) + (in << Divisor)) must not exceed 0xFFFFFFFF.
    //------------------------------------------------------------------------
    template <typename InputType>
    class LowPassFilter
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct with the given @c initialValue.
      //----------------------------------------------------------------------
      LowPassFilter(InputType initialValue, uint8_t fcDividend,
                    LowPassFilterDivisor fcDivisor)
          : _value(initialValue), _dividend(fcDividend), _divisor(fcDivisor)
      {
      }
      
      //----------------------------------------------------------------------
      //!  Put @c value into the filter.  Return the filtered value.
      //----------------------------------------------------------------------
      InputType Filter(InputType value)
      {
        _value *= _dividend;
        _value += (value << _divisor);
        _value -= (_dividend * value);
        _value >>= _divisor;
        return(_value);
      }
      
      //----------------------------------------------------------------------
      //!  Return the filtered value.
      //----------------------------------------------------------------------
      InputType Value() const
      {
        return(_value);
      }
      
    private:
      InputType             _value;
      uint8_t               _dividend;
      LowPassFilterDivisor  _divisor;
      
      LowPassFilter();
    };
    
  }  // namespace Pi

}  // namespace Dwm

#endif  // _DWMPILOWPASSFILTER_HH_
