//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmAuthPeerAuthenticator.hh 11414 2020-12-11 08:36:51Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017, 2018, 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthPeerAuthenticator.hh
//!  \brief Dwm::Auth::PeerAuthenticator class definition
//---------------------------------------------------------------------------

#ifndef _DWMAUTHPEERAUTHENTICATOR_HH_
#define _DWMAUTHPEERAUTHENTICATOR_HH_

#include <chrono>
#include <string>

#include <boost/asio.hpp>

#include "DwmAuthPublicKeysFile.hh"
#include "DwmAuthECDHAgreement.hh"
#include "DwmAuthEd25519Keys.hh"

namespace Dwm {

  namespace Auth {

    //------------------------------------------------------------------------
    //!  A simplified peer authenticator.
    //!  This is a bit tricky under the hood like most security software,
    //!  but also to reduce the number of TCP packets transmitted.  I use
    //!  ECDH to create a shared private key, which is then used to encrypt
    //!  an identity.  The encrypted identity and an SHA256 signature of
    //!  the identity (signed using our RSA private key) and the public
    //!  part of our ECDH exchange are transmitted to the peer.  We receive
    //!  the same from the peer.  We decrypt the peer's identity and check
    //!  that the signed message is valid by using the peer's public key
    //!  contained in our public keys file.  The peer does the same with
    //!  our signed message.
    //------------------------------------------------------------------------
    class PeerAuthenticator
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct from a path to our ed25519 private key and a path to a
      //!  file containing public keys.  We expect the peer's ID (typically
      //!  an email address) to be in the public keys file, along with their
      //!  public key.  If this object is constructed from a server, the
      //!  public key file would typically be an "authorized_keys" file.  If
      //!  constructed from a client, the public key file would typically be
      //!  a "known_services" files.
      //!
      //!  If @c myPrivKeyPath is empty, it is assumed to be
      //!  "~/.dwmauth/id_ed25519".  If @c pubKeysPath is empty, it is assumed
      //!  to be "~/.dwmauth/known_services".
      //----------------------------------------------------------------------
      PeerAuthenticator(const std::string & myPrivKeyPath,
                        const std::string & pubKeysPath);

      //----------------------------------------------------------------------
      //!  Construct from @c myId (normally an email address), a @c keyPair
      //!  and a path to a public keys file.  I need this on iOS where I
      //!  store keys in the keychain instead of in plain files.
      //----------------------------------------------------------------------
      PeerAuthenticator(const std::string & myId,
                        const Ed25519::KeyPair & keyPair,
                        const std::string & pubKeysPath);
      
      //----------------------------------------------------------------------
      //!  Authenticate the peer connected to the socket @c fd.  On success,
      //!  returns true and fills @c theirId with the peer's identifier and
      //!  @c agreedKey with the ECDH agreed shared key which can later be
      //!  used for encryption.  Returns false on failure.  @c agreedKey
      //!  should be 32 bytes in length on success.
      //----------------------------------------------------------------------
      bool Authenticate(int fd, std::string & theirId,
                        std::string & agreedKey);

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      bool Authenticate(boost::asio::ip::tcp::socket & s,
                        std::string & theirId,
                        std::string & agreedKey);
      
    private:
      std::string       _myId;
      Ed25519::KeyPair  _myKeys;
      std::string       _pubKeysPath;
      
      bool ECExchange(int fd, const std::string & peerName,
                      Auth::ECDHAgreement & agreement);
      bool ECExchange(boost::asio::ip::tcp::socket & s,
                      Auth::ECDHAgreement & agreement);
      std::string BuildIDAndSignature(const ECDHAgreement & agreement);
      bool IDAndSigExchange(int fd, const std::string & peerName,
                            const ECDHAgreement & agreement,
                            std::string & theirId,
                            std::string & theirIdCrypted,
                            std::string & theirSig);
      bool IDAndSigExchange(boost::asio::ip::tcp::socket & s,
                            const ECDHAgreement & agreement,
                            std::string & theirId,
                            std::string & theirIdCrypted,
                            std::string & theirSig);
      bool Verify(const std::string & theirId,
                  const std::string & theirIdCrypted,
                  const std::string & theirSig,
                  const std::string & theirPublic) const;
      int BytesReady(int fd) const;
      bool WaitForBytesReady(int fd, uint32_t numBytes,
                             std::chrono::milliseconds timeout) const;
    };
    
  }  // namespace Auth

}  // namespace Dwm

#endif  // _DWMAUTHPEERAUTHENTICATOR_HH_
