//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/trunk/classes/include/DwmDnsRRDataTLSA.hh 10129 $
// @(#) $Id: DwmDnsRRDataTLSA.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataTLSA.hh
//!  \brief Dwm::Dns::RRDataTLSA class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATATLSA_HH_
#define _DWMDNSRRDATATLSA_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates TLSA resource record data.  See RFC6698 for details.
    //------------------------------------------------------------------------
    class RRDataTLSA
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 52;
      
      //----------------------------------------------------------------------
      //!  Construct with an empty certificate association.
      //----------------------------------------------------------------------
      RRDataTLSA();

      //----------------------------------------------------------------------
      //!  Construct with the given @c certificateUsage, @c selector,
      //!  @c matchingType and @c certificateAssociation.
      //----------------------------------------------------------------------
      RRDataTLSA(uint8_t certificateUsage, uint8_t selector,
                 uint8_t matchingType, 
                 const std::string & certificateAssociation);

      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataTLSA & tlsa) const;
      
      //----------------------------------------------------------------------
      //!  Returns the contained certificate usage.
      //----------------------------------------------------------------------
      uint8_t CertificateUsage() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the contained certificate usage.
      //----------------------------------------------------------------------
      uint8_t CertificateUsage(uint8_t certificateUsage);

      //----------------------------------------------------------------------
      //!  Returns the contained selector.
      //----------------------------------------------------------------------
      uint8_t Selector() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the contained selector.
      //----------------------------------------------------------------------
      uint8_t Selector(uint8_t selector);

      //----------------------------------------------------------------------
      //!  Returns the contained matching type.
      //----------------------------------------------------------------------
      uint8_t MatchingType() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained matching type.
      //----------------------------------------------------------------------
      uint8_t MatchingType(uint8_t matchingType);

      //----------------------------------------------------------------------
      //!  Returns the contained certificate association.
      //----------------------------------------------------------------------
      const std::string & CertificateAssociation() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the contained certificate association.
      //----------------------------------------------------------------------
      const std::string &
      CertificateAssociation(const std::string & certificateAssociation);

      //----------------------------------------------------------------------
      //!  Encodes the TLSA resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded TLSA resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded TLSA resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;

      //----------------------------------------------------------------------
      //!  Decodes the TLSA resource record data from buffer @c pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded  TLSA resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt was too short to
      //!  contain an encoded TLSA resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);

      //----------------------------------------------------------------------
      //!  Print the TLSA resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataTLSA & tlsa);
      
    private:
      uint8_t      _certificateUsage;
      uint8_t      _selector;
      uint8_t      _matchingType;
      std::string  _certificateAssociation;
    };
    
  }  // namespace Dns

  
}  // namespace Dwm

#endif  // _DWMDNSRRDATATLSA_HH_
