//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/trunk/classes/include/DwmDnsRRDataDNSKEY.hh 10129 $
// @(#) $Id: DwmDnsRRDataDNSKEY.hh 10129 2018-01-27 04:20:26Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataDNSKEY.hh
//!  \brief Dwm::Dns::RRDataDNSKEY class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSRRDATADNSKEY_HH_
#define _DWMDNSRRDATADNSKEY_HH_

#include <cstdint>
#include <iostream>
#include <string>

#include "DwmDnsRRData.hh"

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  @ingroup rrdatagroup
    //!  Encapsulates DNSKEY resource record data.  See RFC 4034 for details.
    //------------------------------------------------------------------------
    class RRDataDNSKEY
      : public RRData
    {
    public:
      static const uint16_t  k_rrtype = 48;

      static const uint16_t  k_flagsZoneKeyMask          = 0x0080;
      static const uint16_t  k_flagsSecureEntryPointMask = 0x8000;

      static const uint8_t   k_algorithmReserved       = 0;
      static const uint8_t   k_algorithmRSAMD5         = 1;
      static const uint8_t   k_algorithmDiffieHellman  = 2;
      static const uint8_t   k_algorithmDSA            = 3;
      static const uint8_t   k_algorithmECC            = 4;
      static const uint8_t   k_algorithmRSASHA1        = 5;
      static const uint8_t   k_algorithmINDIRECT       = 252;
      static const uint8_t   k_algorithmPRIVATEDNS     = 253;
      static const uint8_t   k_algorithmPRIVATEOID     = 254;
      
      //----------------------------------------------------------------------
      //!  Constructs empty DNSKEY resource record data.
      //----------------------------------------------------------------------
      RRDataDNSKEY();
      
      //----------------------------------------------------------------------
      //!  Construct DNSKEY resource record data with the given @c flags,
      //!  @c protocol, @c algorithm and @c publicKey.
      //----------------------------------------------------------------------
      RRDataDNSKEY(uint16_t flags, uint8_t protocol, uint8_t algorithm,
                   const std::string & publicKey);
      
      //----------------------------------------------------------------------
      //!  Equal-to operator, mostly for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const RRDataDNSKEY & dnskey) const;
      
      //----------------------------------------------------------------------
      //!  Returns the flags in the DNSKEY resource record data.
      //----------------------------------------------------------------------
      uint16_t Flags() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the flags in the DNSKEY resource record data.
      //----------------------------------------------------------------------
      uint16_t Flags(uint16_t flags);

      //----------------------------------------------------------------------
      //!  Per RFC 4034, protocol must _always_ be 3.
      //----------------------------------------------------------------------
      uint8_t Protocol() const;

      //----------------------------------------------------------------------
      //!  Per RFC 4034, protocol must _always_ be 3.
      //----------------------------------------------------------------------
      uint8_t Protocol(uint8_t protocol);
      
      //----------------------------------------------------------------------
      //!  Returns the algorithm in the DNSKEY resource record data.
      //----------------------------------------------------------------------
      uint8_t Algorithm() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the algorithm in the DNSKEY resource record data.
      //----------------------------------------------------------------------
      uint8_t Algorithm(uint8_t algorithm);
      
      //----------------------------------------------------------------------
      //!  Returns the public key in the DNSKEY resource record data.
      //----------------------------------------------------------------------
      const std::string & PublicKey() const;
      
      //----------------------------------------------------------------------
      //!  Set and returns the public key in the DNSKEY resource record data.
      //----------------------------------------------------------------------
      const std::string & PublicKey(const std::string & publicKey);
      
      //----------------------------------------------------------------------
      //!  Encodes the DNSKEY resource record data into buffer @ pkt of length
      //!  @c pktlen, starting at @c ptr.  Returns the address immediately
      //!  following the encoded DNSKEY resource record data in @c pkt on
      //!  success.  Throws std::out_of_range if @c pkt is too short to
      //!  contain the encoded DNSKEY resource record data.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                      LabelPositions & lps) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the DNSKEY resource record data from buffer @c pkt of
      //!  length @c pktlen, starting at @c ptr.  Returns the address
      //!  immediately following the encoded  DNSKEY resource record data
      //!  in @c pkt on success.  Throws std::out_of_range if @c pkt was too
      //!  short to contain an encoded DNSKEY resource record data.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen, uint16_t rdlen);

      //----------------------------------------------------------------------
      //!  Print the DNSKEY resource record data to an ostream in human
      //!  readable form.
      //----------------------------------------------------------------------
      friend std::ostream & operator << (std::ostream & os,
                                         const RRDataDNSKEY & dnskey);
      
    private:
      uint16_t     _flags;
      uint8_t      _protocol;
      uint8_t      _algorithm;
      std::string  _publicKey;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSRRDATADNSKEY_HH_
