//===========================================================================
// @(#) $Name:$
// @(#) $Id: dwmdns.cc 11271 2020-11-23 01:14:19Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file dwmdns.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#include <algorithm>
#include <iostream>

#include "DwmOptArgs.hh"
#include "DwmStringUtils.hh"
#include "DwmSvnTag.hh"
#include "DwmDnsResolver.hh"
#include "DwmDnsUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/trunk/apps/dwmdns/dwmdns.cc 11271 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static uint16_t RRNameToType(std::string rrname)
{
  using RR = Dwm::Dns::ResourceRecord;
  uint16_t  rc = RR::k_typeUNKNOWN;
  static const map<string,uint16_t>  nameToType = {
    { "a",            RR::k_typeA },
    { "ns",           RR::k_typeNS },
    { "cname",        RR::k_typeCNAME },
    { "aaaa",         RR::k_typeAAAA },
    { "soa",          RR::k_typeSOA },
    { "mb",           RR::k_typeMB },
    { "mg",           RR::k_typeMG },
    { "mr",           RR::k_typeMR },
    { "ptr",          RR::k_typePTR },
    { "hinfo",        RR::k_typeHINFO },
    { "minfo",        RR::k_typeMINFO },
    { "mx",           RR::k_typeMX },
    { "txt",          RR::k_typeTXT },
    { "rp",           RR::k_typeRP },
    { "key",          RR::k_typeKEY },
    { "loc",          RR::k_typeLOC },
    { "srv",          RR::k_typeSRV },
    { "cert",         RR::k_typeCERT },
    { "ds",           RR::k_typeDS },
    { "sshfp",        RR::k_typeSSHFP },
    { "rrsig",        RR::k_typeRRSIG },
    { "nsec",         RR::k_typeNSEC },
    { "dnskey",       RR::k_typeDNSKEY },
    { "dhcid",        RR::k_typeDHCID },
    { "nsec3",        RR::k_typeNSEC3 },
    { "nsec3param",   RR::k_typeNSEC3PARAM },
    { "tlsa",         RR::k_typeTLSA },
    { "smimea",       RR::k_typeSMIMEA },
    { "openpgpkey",   RR::k_typeOPENPGPKEY },
    { "uri",          RR::k_typeURI },
    { "caa",          RR::k_typeCAA }
  };
  std::transform(rrname.begin(), rrname.end(), rrname.begin(),
                 [] (unsigned char c) { return std::tolower(c); });
  auto  it = nameToType.find(rrname);
  if (it != nameToType.end()) {
    rc = it->second;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool ShouldUseEDNS(uint16_t rrtype)
{
  using RR = Dwm::Dns::ResourceRecord;

  bool  rc = false;
  switch (rrtype) {
    case RR::k_typeKEY:
    case RR::k_typeCERT:
    case RR::k_typeSSHFP:
    case RR::k_typeRRSIG:
    case RR::k_typeNSEC:
    case RR::k_typeDNSKEY:
    case RR::k_typeNSEC3:
    case RR::k_typeNSEC3PARAM:
    case RR::k_typeTLSA:
    case RR::k_typeSMIMEA:
    case RR::k_typeCDNSKEY:
    case RR::k_typeOPENPGPKEY:
      rc = true;
      break;
    default:
      break;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool GetRRs(Dns::Resolver & resolver, const std::string & name,
                   const std::string & type,
                   vector<Dns::ResourceRecord> & rrs, bool doEDNS = false,
                   bool doDNSSEC = false, bool tcpFallback = true)
{
  using RR = Dwm::Dns::ResourceRecord;
  bool  rc = false;
  uint16_t  rrtype = RRNameToType(type);
  if (rrtype != RR::k_typeUNKNOWN) {
    if (ShouldUseEDNS(rrtype)) {
      doEDNS = true;
    }
    vector<Dns::ResourceRecord>  trrs;
    rc = resolver.Get(name, rrtype, trrs, doEDNS, doDNSSEC, tcpFallback);
    if (! trrs.empty()) {
      for (auto & rr : trrs) {
        rrs.push_back(rr);
      }
    }
  }
  else {
    cerr << "Unknown resource record type " << name << '\n';
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  Dwm::OptArgs  optArgs;
  optArgs.AddOptArg("t:", "types", false, "",
                    "comma-separated list of record types");
  optArgs.AddOptArg("s:", "server", false, "",
                    "name server address");
  optArgs.AddNormalArg("host(s)", true);

  int  nextArg = optArgs.Parse(argc, argv);
  if (nextArg == argc) {
    optArgs.Usage(argv[0]);
    return 1;
  }
  
  string  requestedTypes = optArgs.Get<string>('t');
  if (requestedTypes.empty()) {
    requestedTypes = "aaaa,a";
  }
  vector<string>  rectypes;
  Dwm::StringUtils::StringToVector(requestedTypes, ',', rectypes);
  
  Dns::Resolver                     resolver;
  string  server = optArgs.Get<string>('s');
  if (! server.empty()) {
    try {
      resolver.NameServers().clear();
      resolver.NameServers().push_back(Dns::NameServer(server));
    }
    catch (std::exception & ex) {
      cerr << ex.what() << '\n';
      return 1;
    }
  }
  std::vector<Dns::ResourceRecord>  rrs;

  for (int arg = nextArg; arg < argc; ++arg) {
    string  name = argv[arg];
    if (Dns::ToArpa(argv[arg], name)) {
      GetRRs(resolver, name, "ptr", rrs);
    }
    else {
      for (auto & type : rectypes) {
        GetRRs(resolver, argv[arg], type, rrs);
      }
    }
  }
  for (auto & rr : rrs) {
    cout << rr << '\n';
  }
  return 0;
}
