//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020, 2022
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverWebAppWorker.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::WebAppWorker class implementation
//---------------------------------------------------------------------------

#include <functional>
#include <map>
#include <pugixml.hpp>

#include "DwmSysLogger.hh"
#include "DwmWebUtils.hh"
#include "DwmMcroverWebAppStatusHandler.hh"

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool WordPressResponseOK(WebAppStatusJob & job)
    {
      string url = job.webAppTarget.URI()
        + "/?rest_route=/wp/v2/posts&_fields=author&per_page=5";
      nlohmann::json  json;
      if (WebUtils::GetJson(url, json, job.getFail,
                            job.webAppTarget.ValidateCertificate())) {
        if (! json.is_array()) {
          job.getFail.FailNum(1001);
        }
        else {
          auto it = json.begin();
          for ( ; it != json.end(); ++it) {
            if (! it->is_object()) {
              job.getFail.FailNum(1001);
              break;
            }
            else {
              auto authit = it->find("author");
              if (authit == it->end()) {
                job.getFail.FailNum(1001);
                break;
              }
              if (! authit->is_number()) {
                job.getFail.FailNum(1001);
                break;
              }
            }
          }
          if (it == json.end()) {
            job.getFail.FailNum(WebUtils::GetFailure::k_failNumNone);
          }
        }
      }
      return (WebUtils::GetFailure::k_failNumNone == job.getFail.FailNum());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool PiwigoResponseOK(WebAppStatusJob & job)
    {
      string url = job.webAppTarget.URI()
        + "/ws.php?format=json&method=pwg.categories.getList";
      nlohmann::json  json;
      if (WebUtils::GetJson(url, json, job.getFail,
                            job.webAppTarget.ValidateCertificate())) {
        if (! json.is_object()) {
          Syslog(LOG_ERR, "JSON is not an object");
          job.getFail.FailNum(1001);
        }
        else {
          auto  result = json.find("result");
          if ((result != json.end()) && result->is_object()) {
            auto  status = json.find("stat");
            if ((status != json.end()) && status->is_string()) {
              if (status->get<string>() == "ok") {
                auto categories = result->find("categories");
                if ((categories != result->end()) && categories->is_array()) {
                  auto catit = categories->begin();
                  for ( ; catit != categories->end(); ++catit) {
                    auto caturl = catit->find("url");
                    if (caturl != catit->end()) {
                      if (! caturl->is_string()) {
                        Syslog(LOG_ERR,
                               "'result/categories[N]/url' is not a string");
                        break;
                      }
                    }
                    else {
                      Syslog(LOG_ERR,
                             "url not found in 'result/categories[N]'");
                      break;
                    }
                  }
                  if (catit == categories->end()) {
                    job.getFail.FailNum(WebUtils::GetFailure::k_failNumNone);
                  }
                }
                else {
                  Syslog(LOG_ERR, "'result/categories' not found");
                  job.getFail.FailNum(1001);
                }
              }
              else {
                Syslog(LOG_ERR, "status: %s", status->get<string>().c_str());
                job.getFail.FailNum(1001);
              }
            }
            else {
              Syslog(LOG_ERR, "'stat' not found");
              job.getFail.FailNum(1001);
            }
          }
          else {
            Syslog(LOG_ERR, "'result' not found");
            job.getFail.FailNum(1001);
          }
        }
      }
      return (WebUtils::GetFailure::k_failNumNone == job.getFail.FailNum());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool XMLQueryOK(const std::string & xml,
                           const std::string & queryStr)
    {
      bool                    rc = false;
      pugi::xml_document      doc;
      pugi::xml_parse_result  parseResult =
        doc.load_buffer(xml.data(), xml.size());
      if (parseResult) {
        try {
          pugi::xpath_node_set  xmlNodes = doc.select_nodes(queryStr.c_str());
          rc = (! xmlNodes.empty());
        }
        catch (const pugi::xpath_exception & xpex) {
          Syslog(LOG_ERR, "xpath_exception");
        }
        catch (...) {
          Syslog(LOG_ERR, "Unknown exception");
        }
      }
      else {
        Syslog(LOG_ERR, "Failed to parse XML");
      }
      
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool PlexResponseOK(WebAppStatusJob & job)
    {
      string url = job.webAppTarget.URI();
      if (! job.webAppTarget.Params().empty()) {
        url += '?';
        auto  paramit = job.webAppTarget.Params().begin();
        url += *paramit;
        ++paramit;
        for ( ; paramit != job.webAppTarget.Params().end(); ++paramit) {
          url += '&';
          url += *paramit;
        }
      }
      namespace http = boost::beast::http;
      http::response<http::string_body>  response;
      if (WebUtils::GetResponse(url, response, job.getFail,
                                job.webAppTarget.ValidateCertificate())) {
        if (! XMLQueryOK(response.body(), job.webAppTarget.Xpath())) {
          job.getFail.FailNum(1001);
          Syslog(LOG_ERR, "Plex XML query failed for '%s' at %s",
                 job.webAppTarget.Xpath().c_str(),
                 job.webAppTarget.URI().c_str());
        }
        else {
          Syslog(LOG_INFO, "Plex XML query passed for '%s' at %s",
                 job.webAppTarget.Xpath().c_str(),
                 job.webAppTarget.URI().c_str());
        }
      }
      return (WebUtils::GetFailure::k_failNumNone == job.getFail.FailNum());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static const map<string,function<bool(WebAppStatusJob &)>>
    sg_appTests = {
      { "wordpress", WordPressResponseOK },
      { "piwigo",    PiwigoResponseOK },
      { "plex",      PlexResponseOK }
    };
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    WebAppWorker::WebAppWorker(WebAppStatusHandler & statusHandler)
        : _statusHandler(statusHandler)
    {
      Start();
      Syslog(LOG_DEBUG, "WebAppWorker %p created", this);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    WebAppWorker::~WebAppWorker()
    {
      Syslog(LOG_DEBUG, "WebAppWorker %p destroyed", this);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WebAppWorker::ProcessWork(WebAppStatusJob & job)
    {
      Syslog(LOG_DEBUG, "Processing job %s, %s",
             ((string)(job.addr)).c_str(), job.webAppTarget.URI().c_str());
      auto  appti = sg_appTests.find(job.webAppTarget.AppName());
      if (appti != sg_appTests.end()) {
        if (appti->second(job)) {
          Syslog(LOG_INFO, "%s passed for %s", appti->first.c_str(),
                 job.webAppTarget.URI().c_str());
        }
        else {
          Syslog(LOG_ERR, "%s failed for %s", appti->first.c_str(),
                 job.webAppTarget.URI().c_str());
        }
      }
      else {
        Syslog(LOG_ERR, "No test for app '%s'",
               job.webAppTarget.AppName().c_str());
        job.getFail.FailNum(1001);
      }
      _statusHandler.HandleStatus(job);
      return;
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm
