//===========================================================================
//  Copyright (c) Daniel W. McRobb 2022
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverWebAppAlert.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::WebAppAlert class implementation
//---------------------------------------------------------------------------

#include <boost/algorithm/string.hpp>

#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmIOUtils.hh"
#include "DwmStreamIO.hh"
#include "DwmIpv4Address.hh"
#include "DwmSvnTag.hh"
#include "DwmMcroverWebAppAlert.hh"
#include "DwmMcroverUtils.hh"

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    WebAppAlert::WebAppAlert()
        : _uri(), _failType("NONE"), _appName("APP")
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    WebAppAlert::WebAppAlert(const string & uri,
                             const string & failType,
                             const string & appName)
        : _uri(uri), _failType(failType), _appName(appName)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & WebAppAlert::URI() const
    {
      return _uri;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & WebAppAlert::URI(const string & uri)
    {
      _uri = uri;
      return _uri;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & WebAppAlert::FailType() const
    {
      return _failType;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & WebAppAlert::FailType(const string & failType)
    {
      return _failType = failType;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & WebAppAlert::AppName() const
    {
      return _appName;
    }
  
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & WebAppAlert::AppName(const string & appName)
    {
      return _appName = appName;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string WebAppAlert::DisplayString() const
    {
      string  rc = _uri;
      if (_failType != _appName) {
        rc += ':' + _failType;
      }
      rc += ' ' + boost::to_upper_copy<std::string>(_appName);
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json WebAppAlert::ToJson() const
    {
      nlohmann::json  j;
      j["uri"] = _uri;
      j["failType"] = _failType;
      j["appName"] = _appName;
      return j;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool WebAppAlert::FromJson(const nlohmann::json & j)
    {
      bool  rc = false;
      if (j.is_object()) {
        auto  uri = j.find("uri");
        if ((uri != j.end()) && uri->is_string()) {
          _uri = uri->get<string>();
          auto  failType = j.find("failType");
          if ((failType != j.end()) && failType->is_string()) {
            _failType = failType->get<string>();
            auto  appName = j.find("appName");
            if ((appName != j.end()) && appName->is_string()) {
              _appName = appName->get<string>();
              rc = true;
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    istream & WebAppAlert::Read(istream & is)
    {
      if (Dwm::StreamIO::Read(is, _uri)) {
        if (Dwm::StreamIO::Read(is, _failType)) {
          Dwm::StreamIO::Read(is, _appName);
        }
      }
      return is;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t WebAppAlert::Read(int fd)
    {
      ssize_t  rc = -1;
      ssize_t  bytesRead = Dwm::DescriptorIO::Read(fd, _uri);
      if (0 < bytesRead) {
        rc = bytesRead;
        bytesRead = Dwm::DescriptorIO::Read(fd, _failType);
        if (0 < bytesRead) {
          rc += bytesRead;
          bytesRead = Dwm::DescriptorIO::Read(fd, _appName);
          if (0 < bytesRead) {
            rc += bytesRead;
          }
          else {
            rc = -1;
          }
        }
        else {
          rc = -1;
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t WebAppAlert::Read(FILE * f)
    {
      size_t  rc = 0;
      if (Dwm::FileIO::Read(f, _uri)) {
        if (Dwm::FileIO::Read(f, _failType)) {
          rc = Dwm::FileIO::Read(f, _appName);
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & WebAppAlert::Write(ostream & os) const
    {
      if (Dwm::StreamIO::Write(os, _uri)) {
        if (Dwm::StreamIO::Write(os, _failType)) {
          Dwm::StreamIO::Write(os, _appName);
        }
      }
      return os;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t WebAppAlert::Write(int fd) const
    {
      ssize_t  rc = -1;
      ssize_t  bytesWritten = Dwm::DescriptorIO::Write(fd, _uri);
      if (0 < bytesWritten) {
        rc = bytesWritten;
        bytesWritten = Dwm::DescriptorIO::Write(fd, _failType);
        if (0 < bytesWritten) {
          rc += bytesWritten;
          bytesWritten = Dwm::DescriptorIO::Write(fd, _appName);
          if (0 < bytesWritten) {
            rc += bytesWritten;
          }
          else {
            rc = -1;
          }
        }
        else {
          rc = -1;
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t WebAppAlert::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (Dwm::FileIO::Write(f, _uri)) {
        if (Dwm::FileIO::Write(f, _failType)) {
          rc = Dwm::FileIO::Write(f, _appName);
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t WebAppAlert::StreamedLength() const
    {
      return (Dwm::IOUtils::StreamedLength(_uri)
              + Dwm::IOUtils::StreamedLength(_failType)
              + Dwm::IOUtils::StreamedLength(_appName));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool WebAppAlert::operator == (const WebAppAlert & alert) const
    {
      return ((_uri == alert._uri)
              && (_failType == alert._failType)
              && (_appName == alert._appName));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool WebAppAlert::operator < (const WebAppAlert & alert) const
    {
      bool  rc = false;
      if (_uri < alert._uri) {
        rc = true;
      }
      else if (_uri == alert._uri) {
        if (_failType < alert._failType) {
          rc = true;
        }
        else if (_failType == alert._failType) {
          rc = (_appName < alert._appName);
        }
      }
      return rc;
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm
