//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/classes/src/DwmMcroverTcp6DstAddr.cc 11949 $
// @(#) $Id: DwmMcroverTcp6DstAddr.cc 11949 2022-04-17 17:04:51Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverTcp6DstAddr.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::Tcp6DstAddr class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>  // for htons() and friends
  #include <unistd.h>
}

#include <cstring>  // for memset()
#include <stdexcept>

#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmIOUtils.hh"
#include "DwmStreamIO.hh"
#include "DwmSvnTag.hh"
#include "DwmMcroverUtils.hh"
#include "DwmMcroverTcp6DstAddr.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/classes/src/DwmMcroverTcp6DstAddr.cc 11949 $");

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Tcp6DstAddr::Tcp6DstAddr()
        : _name()
    {
      memset(&_sockAddr, 0, sizeof(_sockAddr));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Tcp6DstAddr::Tcp6DstAddr(const std::string & name, Ipv6Address addr,
                             uint16_t port)
    {
      Set(name, addr, port);
    }
    

#if 0
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Tcp6DstAddr::Tcp6DstAddr(Ipv6Address, uint16_t port)
    {
      Set(addr, port);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Tcp6DstAddr::Tcp6DstAddr(const std::string & nameOrAddr, uint16_t port)
    {
      in_addr_t  addr = Utils::GetHostIpv6Addr(nameOrAddr);
      if (INADDR_NONE == addr) {
        throw std::invalid_argument("invalid IPv6 hostname or address");
      }
      Set(addr, port);
    }
#endif
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Tcp6DstAddr::Set(const std::string & name, Ipv6Address addr,
                          uint16_t port)
    {
      _name = name;
      memset(&_sockAddr, 0, sizeof(_sockAddr));
#ifndef __linux__
      _sockAddr.sin6_len = sizeof(_sockAddr);
#endif
      _sockAddr.sin6_family = AF_INET6;
      _sockAddr.sin6_addr = addr;
      _sockAddr.sin6_port = htons(port);
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const struct sockaddr *Tcp6DstAddr::SockAddr() const
    {
      return (const struct sockaddr *)&_sockAddr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string	& Tcp6DstAddr::Name() const
    {
      return _name;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string	& Tcp6DstAddr::Name(const std::string & name)
    {
      _name = name;
      return _name;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Ipv6Address Tcp6DstAddr::Address() const
    {
      return Ipv6Address(_sockAddr.sin6_addr);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Ipv6Address Tcp6DstAddr::Address(const Ipv6Address & addr)
    {
      _sockAddr.sin6_addr = addr;
      return Ipv6Address(_sockAddr.sin6_addr);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t Tcp6DstAddr::Port() const
    {
      return ntohs(_sockAddr.sin6_port);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t Tcp6DstAddr::Port(uint16_t port)
    {
      _sockAddr.sin6_port = htons(port);
      return ntohs(_sockAddr.sin6_port);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Tcp6DstAddr::operator < (const Tcp6DstAddr & dstAddr) const
    {
      bool  rc = false;
      if (_name < dstAddr._name) {
        rc = true;
      }
      else if (_name == dstAddr._name) {
        Ipv6Address  myAddr6(_sockAddr.sin6_addr);
        Ipv6Address  dstAddr6(dstAddr._sockAddr.sin6_addr);
        if (myAddr6 < dstAddr6) {
          rc = true;
        }
        else if (myAddr6 == dstAddr6) {
          if (ntohs(_sockAddr.sin6_port)
              < ntohs(dstAddr._sockAddr.sin6_port)) {
            rc = true;
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & Tcp6DstAddr::Read(std::istream & is)
    {
      if (is) {
        if (StreamIO::Read(is, _name)) {
          Ipv6Address  addr;
          if (addr.Read(is)) {
            _sockAddr.sin6_addr = addr;
            is.read((char *)&_sockAddr.sin6_port, sizeof(_sockAddr.sin6_port));
          }
        }
      }
      return is;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t Tcp6DstAddr::Read(int fd)
    {
      int  rc = -1;
      if (fd >= 0) {
        int  bytesRead = DescriptorIO::Read(fd, _name);
        if (0 < bytesRead) {
          rc = bytesRead;
          Ipv6Address  addr;
          bytesRead = addr.Read(fd);
          if (bytesRead == sizeof(_sockAddr.sin6_addr)) {
            rc += bytesRead;
            _sockAddr.sin6_addr = addr;
            bytesRead = read(fd, &_sockAddr.sin6_port,
                             sizeof(_sockAddr.sin6_port));
            if (sizeof(_sockAddr.sin6_port) == bytesRead) {
              rc += bytesRead;
            }
            else {
              rc = -1;
            }
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t Tcp6DstAddr::Read(FILE * f)
    {
      size_t  rc = 0;
      
      if (f) {
        if (FileIO::Read(f, _name)) {
          Ipv6Address  addr;
          if (addr.Read(f)) {
            _sockAddr.sin6_addr = addr;
            if (fread(&_sockAddr.sin6_port,
                      sizeof(_sockAddr.sin6_port), 1, f)) {
              rc = 1;
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & Tcp6DstAddr::Write(std::ostream & os) const
    {
      if (os) {
        if (StreamIO::Write(os, _name)) {
          Ipv6Address addr(_sockAddr.sin6_addr);
          if (addr.Write(os)) {
            os.write((const char *)&_sockAddr.sin6_port,
                     sizeof(_sockAddr.sin6_port));
          }
        }
      }
      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t Tcp6DstAddr::Write(int fd) const
    {
      ssize_t  rc = -1;
      if (fd >= 0) {
        ssize_t  bytesWritten = DescriptorIO::Write(fd, _name);
        if (bytesWritten > 0) {
          rc = bytesWritten;
          Ipv6Address addr(_sockAddr.sin6_addr);
          bytesWritten = addr.Write(fd);
          if (bytesWritten == sizeof(_sockAddr.sin6_addr)) {
            rc += bytesWritten;
            if (write(fd, (const char *)&_sockAddr.sin6_port,
                      sizeof(_sockAddr.sin6_port))
                == sizeof(_sockAddr.sin6_port)) {
              rc += sizeof(_sockAddr.sin6_port);
            }
            else {
              rc = -1;
            }
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t Tcp6DstAddr::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (f) {
        if (FileIO::Write(f, _name)) {
          Ipv6Address addr(_sockAddr.sin6_addr);
          if (addr.Write(f)) {
            if (fwrite((const char *)&_sockAddr.sin6_port,
                       sizeof(_sockAddr.sin6_port), 1, f)) {
              rc = 1;
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t Tcp6DstAddr::StreamedLength() const
    {
      return (sizeof(_sockAddr) + IOUtils::StreamedLength(_name));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json Tcp6DstAddr::ToJson() const
    {
      nlohmann::json  j;
      j["name"] = _name;
      char  buf[64] = { '\0' };
      inet_ntop(AF_INET6, &_sockAddr.sin6_addr, buf, sizeof(buf));
      j["addr"] = buf;
      j["port"] = ntohs(_sockAddr.sin6_port);
      return j;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Tcp6DstAddr::FromJson(const nlohmann::json & json)
    {
      bool  rc = false;
      if (json.is_object()) {
        auto  name = json.find("name");
        if ((name != json.end()) && name->is_string()) {
          _name = name->get<std::string>();
          auto  addr = json.find("addr");
          if ((addr != json.end()) && addr->is_string()) {
            if (inet_pton(AF_INET6, addr->get<std::string>().c_str(),
                          &_sockAddr.sin6_addr) == 1) {
              auto  port = json.find("port");
              if ((port != json.end()) && port->is_number()) {
                _sockAddr.sin6_port = htons(port->get<uint16_t>());
                rc = true;
              }
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Tcp6DstAddr::operator == (const Tcp6DstAddr & dst) const
    {
      return ((_name == dst._name)
              && (Ipv6Address(_sockAddr.sin6_addr)
                  == Ipv6Address(dst._sockAddr.sin6_addr))
              && (_sockAddr.sin6_port == dst._sockAddr.sin6_port));
    }

  }  // namespace Mcrover

}  // namespace Dwm
