//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/classes/src/DwmMcroverTcp4DstAddr.cc 11886 $
// @(#) $Id: DwmMcroverTcp4DstAddr.cc 11886 2021-04-07 05:16:39Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverTcp4DstAddr.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::Tcp4DstAddr class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>  // for htons() and friends
  #include <unistd.h>
}

#include <cstring>  // for memset()
#include <stdexcept>

#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmIOUtils.hh"
#include "DwmStreamIO.hh"
#include "DwmSvnTag.hh"
#include "DwmMcroverUtils.hh"
#include "DwmMcroverTcp4DstAddr.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/classes/src/DwmMcroverTcp4DstAddr.cc 11886 $");

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Tcp4DstAddr::Tcp4DstAddr()
        : _name()
    {
      memset(&_sockAddr, 0, sizeof(_sockAddr));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Tcp4DstAddr::Tcp4DstAddr(const std::string & name, uint32_t addr,
                             uint16_t port)
    {
      Set(name, addr, port);
    }
    

#if 0
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Tcp4DstAddr::Tcp4DstAddr(uint32_t addr, uint16_t port)
    {
      Set(addr, port);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Tcp4DstAddr::Tcp4DstAddr(const std::string & nameOrAddr, uint16_t port)
    {
      in_addr_t  addr = Utils::GetHostIpv4Addr(nameOrAddr);
      if (INADDR_NONE == addr) {
        throw std::invalid_argument("invalid IPv4 hostname or address");
      }
      Set(addr, port);
    }
#endif
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Tcp4DstAddr::Set(const std::string & name, uint32_t addr,
                          uint16_t port)
    {
      _name = name;
      memset(&_sockAddr, 0, sizeof(_sockAddr));
#ifndef __linux__
      _sockAddr.sin_len = sizeof(_sockAddr);
#endif
      _sockAddr.sin_family = AF_INET;
      _sockAddr.sin_addr.s_addr = addr;
      _sockAddr.sin_port = htons(port);
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const struct sockaddr *Tcp4DstAddr::SockAddr() const
    {
      return (const struct sockaddr *)&_sockAddr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string	& Tcp4DstAddr::Name() const
    {
      return _name;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string	& Tcp4DstAddr::Name(const std::string & name)
    {
      _name = name;
      return _name;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    in_addr_t Tcp4DstAddr::Address() const
    {
      return _sockAddr.sin_addr.s_addr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    in_addr_t Tcp4DstAddr::Address(in_addr_t addr)
    {
      _sockAddr.sin_addr.s_addr = addr;
      return _sockAddr.sin_addr.s_addr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t Tcp4DstAddr::Port() const
    {
      return ntohs(_sockAddr.sin_port);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t Tcp4DstAddr::Port(uint16_t port)
    {
      _sockAddr.sin_port = htons(port);
      return ntohs(_sockAddr.sin_port);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Tcp4DstAddr::operator < (const Tcp4DstAddr & dstAddr) const
    {
      bool  rc = false;
      if (_name < dstAddr._name) {
        rc = true;
      }
      else if (_name == dstAddr._name) {
        if (ntohl(_sockAddr.sin_addr.s_addr)
            < ntohl(dstAddr._sockAddr.sin_addr.s_addr)) {
          rc = true;
        }
        else if (_sockAddr.sin_addr.s_addr
                 == dstAddr._sockAddr.sin_addr.s_addr) {
          if (ntohs(_sockAddr.sin_port) < ntohs(dstAddr._sockAddr.sin_port)) {
            rc = true;
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & Tcp4DstAddr::Read(std::istream & is)
    {
      if (is) {
        if (StreamIO::Read(is, _name)) {
          if (is.read((char *)&_sockAddr.sin_addr.s_addr,
                      sizeof(_sockAddr.sin_addr.s_addr))) {
            is.read((char *)&_sockAddr.sin_port, sizeof(_sockAddr.sin_port));
          }
        }
      }
      return is;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t Tcp4DstAddr::Read(int fd)
    {
      int  rc = -1;
      if (fd >= 0) {
        int  bytesRead = DescriptorIO::Read(fd, _name);
        if (0 < bytesRead) {
          rc = bytesRead;
          bytesRead = read(fd, &_sockAddr.sin_addr.s_addr,
                           sizeof(_sockAddr.sin_addr.s_addr));
          if (sizeof(_sockAddr.sin_addr.s_addr) == bytesRead) {
            rc += bytesRead;
            bytesRead = read(fd, &_sockAddr.sin_port,
                             sizeof(_sockAddr.sin_port));
            if (sizeof(_sockAddr.sin_port) == bytesRead) {
              rc += bytesRead;
            }
            else {
              rc = -1;
            }
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t Tcp4DstAddr::Read(FILE * f)
    {
      size_t  rc = 0;
      
      if (f) {
        if (FileIO::Read(f, _name)) {
          if (fread(&_sockAddr.sin_addr.s_addr,
                    sizeof(_sockAddr.sin_addr.s_addr), 1, f)) {
            if (fread(&_sockAddr.sin_port, sizeof(_sockAddr.sin_port), 1, f)) {
              rc = 1;
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & Tcp4DstAddr::Write(std::ostream & os) const
    {
      if (os) {
        if (StreamIO::Write(os, _name)) {
          if (os.write((const char *)&_sockAddr.sin_addr.s_addr,
                       sizeof(_sockAddr.sin_addr.s_addr))) {
            os.write((const char *)&_sockAddr.sin_port,
                     sizeof(_sockAddr.sin_port));
          }
        }
      }
      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t Tcp4DstAddr::Write(int fd) const
    {
      ssize_t  rc = -1;
      if (fd >= 0) {
        ssize_t  bytesWritten = DescriptorIO::Write(fd, _name);
        if (bytesWritten > 0) {
          rc = bytesWritten;
          if (write(fd, (const char *)&_sockAddr.sin_addr.s_addr,
                    sizeof(_sockAddr.sin_addr.s_addr))
              == sizeof(_sockAddr.sin_addr.s_addr)) {
            if (write(fd, (const char *)&_sockAddr.sin_port,
                      sizeof(_sockAddr.sin_port))
                == sizeof(_sockAddr.sin_port)) {
              rc += 6;
            }
            else {
              rc = -1;
            }
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t Tcp4DstAddr::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (f) {
        if (FileIO::Write(f, _name)) {
          if (fwrite((const char *)&_sockAddr.sin_addr.s_addr,	
                     sizeof(_sockAddr.sin_addr.s_addr), 1, f)) {
            if (fwrite((const char *)&_sockAddr.sin_port,
                       sizeof(_sockAddr.sin_port), 1, f)) {
              rc = 1;
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t Tcp4DstAddr::StreamedLength() const
    {
      return (6 + IOUtils::StreamedLength(_name));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json Tcp4DstAddr::ToJson() const
    {
      nlohmann::json  j;
      j["name"] = _name;
      j["addr"] = inet_ntoa(_sockAddr.sin_addr);
      j["port"] = ntohs(_sockAddr.sin_port);
      return j;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Tcp4DstAddr::FromJson(const nlohmann::json & json)
    {
      bool  rc = false;
      if (json.is_object()) {
        auto  name = json.find("name");
        if ((name != json.end()) && name->is_string()) {
          _name = name->get<std::string>();
          auto  addr = json.find("addr");
          if ((addr != json.end()) && addr->is_string()) {
            _sockAddr.sin_addr.s_addr =
              inet_addr(addr->get<std::string>().c_str());
            auto  port = json.find("port");
            if ((port != json.end()) && port->is_number()) {
              _sockAddr.sin_port = htons(port->get<uint16_t>());
              rc = true;
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Tcp4DstAddr::operator == (const Tcp4DstAddr & dst) const
    {
      return ((_name == dst._name)
              && (_sockAddr.sin_addr.s_addr == dst._sockAddr.sin_addr.s_addr)
              && (_sockAddr.sin_port == dst._sockAddr.sin_port));
    }

  }  // namespace Mcrover

}  // namespace Dwm
