//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/classes/src/DwmMcroverTCPAddress.cc 11886 $
// @(#) $Id: DwmMcroverTCPAddress.cc 11886 2021-04-07 05:16:39Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverTCPAddress.cc
//!  \author Daniel W. McRobb
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
}

#include "DwmMcroverTCPAddress.hh"
#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmStreamIO.hh"
#include "DwmIOUtils.hh"

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    TCPAddress::TCPAddress(const Ipv6Address & addr, uint16_t port)
        : _addr(addr), _port(port)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    TCPAddress::TCPAddress(const Ipv4Address & addr, uint16_t port)
        : _addr(addr), _port(port)
    {}

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    TCPAddress::TCPAddress(const std::string & addr, uint16_t port)
    {
      if (addr == "inaddr_any") {
        _addr = Ipv4Address(INADDR_ANY);
      }
      else if (addr == "in6addr_any") {
        _addr = Ipv6Address(in6addr_any);
      }
      else {
        struct in6_addr  in6Addr;
        if (inet_pton(AF_INET6, addr.c_str(), &in6Addr) == 1) {
          _addr = Ipv6Address(in6Addr);
        }
        else {
          struct in_addr  inAddr;
          if (inet_pton(AF_INET, addr.c_str(), &inAddr) == 1) {
            _addr = Ipv4Address(inAddr.s_addr);
          }
          else {
            _addr = Ipv4Address(INADDR_NONE);
          }
        }
      }
      _port = port;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    TCPAddress::operator std::string () const
    {
      std::string   rc = (std::string)_addr + ':';
      rc += std::to_string(_port);
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool TCPAddress::operator < (const TCPAddress & addr) const
    {
      bool  rc = false;
      if (_addr < addr._addr) {
        rc = true;
      }
      else if (_addr == addr._addr) {
        rc = (_port < addr._port);
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool TCPAddress::operator == (const TCPAddress & addr) const
    {
      return ((_addr == addr._addr) && (_port == addr._port));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & TCPAddress::Read(std::istream & is)
    {
      if (is) {
        if (StreamIO::Read(is, _addr)) {
          StreamIO::Read(is, _port);
        }
      }
      return is;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t TCPAddress::Read(int fd)
    {
      ssize_t  rc = -1;
      if (0 <= fd) {
        ssize_t  bytesRead = DescriptorIO::Read(fd, _addr);
        if (0 < bytesRead) {
          rc = bytesRead;
          bytesRead = DescriptorIO::Read(fd, _port);
          if (sizeof(_port) == bytesRead) {
            rc += bytesRead;
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t TCPAddress::Read(FILE * f)
    {
      size_t  rc = 0;
      if (f) {
        if (FileIO::Read(f, _addr)) {
          if (FileIO::Read(f, _port)) {
            rc = 1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & TCPAddress::Write(std::ostream & os) const
    {
      if (os) {
        if (StreamIO::Write(os, _addr)) {
          StreamIO::Write(os, _port);
        }
      }
      return os;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t TCPAddress::Write(int fd) const
    {
      ssize_t  rc = -1;
      if (0 <= fd) {
        ssize_t  bytesWritten = DescriptorIO::Write(fd, _addr);
        if (0 < bytesWritten) {
          rc = bytesWritten;
          bytesWritten = DescriptorIO::Write(fd, _port);
          if (sizeof(_port) == bytesWritten) {
            rc += bytesWritten;
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t TCPAddress::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (f) {
        if (FileIO::Write(f, _addr)) {
          if (FileIO::Write(f, _port)) {
            rc = 1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t TCPAddress::StreamedLength() const
    {
      return (IOUtils::StreamedLength(_addr) + IOUtils::StreamedLength(_port));
    }
    
  }  // namespace Mcrover

}  // namespace Dwm
