//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/classes/src/DwmMcroverFilesystemUtils.cc 12022 $
// @(#) $Id: DwmMcroverFilesystemUtils.cc 12022 2022-05-30 19:03:23Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverFilesystemUtils.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::FilesystemUtils class implementation
//---------------------------------------------------------------------------

extern "C" {
#ifdef __linux__
  #include <mntent.h>
  #include <sys/vfs.h>
#else
  #include <sys/param.h>
  #include <sys/ucred.h>
  #include <sys/mount.h>
#endif
}

#include <cmath>
#include <cstdio>
#include <cstring>
#include <map>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverFilesystemUtils.hh"
#include "DwmMcroverUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/classes/src/DwmMcroverFilesystemUtils.cc 12022 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool FilesystemUtils::GetAlerts(const AlertOrigin & origin,
                                    const Config & config,
                                    AlertBowl & alerts, bool clearAtEntry)
    {
      if (clearAtEntry) {
        alerts.Clear();
      }
      if (! config.Local().Filesystems().empty()) {
        map<string,pair<bool,uint8_t>>  results;
        for (const auto & cfs : config.Local().Filesystems()) {
          results[cfs.first] = { false, 0 };
        }
        CheckFilesystems(results);
        Ipv4Address  localAddr(Utils::ThisHostIpv4Addr());
        for (const auto result : results) {
          if (! result.second.first) {
            alerts.Add(origin,
                       FilesystemAlert(localAddr, result.first),
                       time((time_t *)0));
            Syslog(LOG_ERR, "Filesystem failed for '%s'",
                   result.first.c_str());
          }
          else {
            auto  cfgit = config.Local().Filesystems().find(result.first);
            if (config.Local().Filesystems().end() != cfgit) {
              if (result.second.second >= cfgit->second) {
                alerts.Add(origin, FilesystemAlert(localAddr,
                                                   result.first,
                                                   result.second.second),
                           time((time_t *)0));
                Syslog(LOG_ERR, "Filesystem capacity failed for '%s'"
                       " (%hhu%% >= %hhu%%)", result.first.c_str(),
                       result.second.second, cfgit->second);
              }
              else {
                Syslog(LOG_INFO, "Filesystem capacity passed for '%s'"
                       " (%hhu%% < %hhu%%)", result.first.c_str(),
                       result.second.second, cfgit->second);
              }
            }
            else {
              Syslog(LOG_INFO, "Filesystem passed for '%s'",
                     result.first.c_str());
            }
          }
        }
      }
      return (! alerts.Empty());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string FilesystemUtils::SanitizedMountPoint(const string & mountPoint)
    {
      string  rc = mountPoint;
      while (rc.size() > 1) {
        if ('/' != rc.back()) {  break;  }
        else  {  rc.pop_back();  }
      }
      return rc;
    }
    
#ifdef __linux__

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    pair<bool,uint8_t>
    FilesystemUtils::CheckFilesystem(const string & mountPoint)
    {
      pair<bool,uint8_t>  rc(false, 0);
      FILE  *f = setmntent("/etc/mtab", "r");
      if (f) {
        string  sanitizedMount = SanitizedMountPoint(mountPoint);
        while (struct mntent *entp = getmntent(f)) {
          if (sanitizedMount == entp->mnt_dir) {
            rc.first = true;
            break;
          }
        }
        endmntent(f);
        if (rc.first) {
          struct statfs  buf;
          if (statfs(sanitizedMount.c_str(), &buf) == 0) {
            auto  used = buf.f_blocks - buf.f_bfree;
            auto  availblks = buf.f_bavail + used;
            rc.second = std::lround((double)used / (double)availblks * 100.0);
          }
          else {
            Syslog(LOG_ERR, "statfs(%s,%p) failed", sanitizedMount.c_str(), &buf);
          }
        }
        else {
          Syslog(LOG_ERR, "Did not find '%s' mount point",
                 sanitizedMount.c_str());
        }
      }
      return rc;
    }
    
#else
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    pair<bool,uint8_t>
    FilesystemUtils::CheckFilesystem(const std::string & mountPoint)
    {
      pair<bool,uint8_t>  rc(false, 0);
      
      int  numfs = getfsstat(nullptr, 0, MNT_WAIT);
      if (numfs > 0) {
        size_t  numBytes = numfs * sizeof(struct statfs);
        struct statfs  *buf = (struct statfs *)calloc(1, numBytes);
        if (buf) {
          string  sanitizedMount = SanitizedMountPoint(mountPoint);
          numfs = getfsstat(buf, numBytes, MNT_WAIT);
          int  i = 0;
          for ( ; i < numfs; ++i) {
            if (sanitizedMount == buf[i].f_mntonname) {
              rc.first = true;
              auto  used = buf[i].f_blocks - buf[i].f_bfree;
              auto  availblks = buf[i].f_bavail + used;
              rc.second = std::lround((double)used / (double)availblks * 100.0);
              break;
            }
          }
          free(buf);
        }
        else {
          Syslog(LOG_ERR, "Failed to allocate %lu bytes", numBytes);
        }
      }
      return rc;
    }
    
#endif
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void
    FilesystemUtils::CheckFilesystems(map<string,pair<bool,uint8_t>> & fss)
    {
      for (auto & fs : fss) {
        fs.second = CheckFilesystem(fs.first);
      }
      return;
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm
