//===========================================================================
//  Copyright (c) Daniel W. McRobb 2022, 2023
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverWebAppTarget.hh
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::WebAppTarget class declaration
//---------------------------------------------------------------------------

#ifndef _DWMMCROVERWEBAPPTARGET_HH_
#define _DWMMCROVERWEBAPPTARGET_HH_

#include <string>
#include <vector>
#include <nlohmann/json.hpp>

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  Encapsulates configuration for a web application target.
    //------------------------------------------------------------------------
    class WebAppTarget
    {
    public:
      //----------------------------------------------------------------------
      //!  Default constructor.
      //----------------------------------------------------------------------
      WebAppTarget();

      //----------------------------------------------------------------------
      //!  Copy constructor.
      //----------------------------------------------------------------------
      WebAppTarget(const WebAppTarget &) = default;
      
      //----------------------------------------------------------------------
      //!  Returns the URI of the target.
      //----------------------------------------------------------------------
      const std::string & URI() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the URI of the target.
      //----------------------------------------------------------------------
      const std::string & URI(const std::string & uri);

      //----------------------------------------------------------------------
      //!  Returns the target application's name.
      //----------------------------------------------------------------------
      const std::string & AppName() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the target application's name.
      //----------------------------------------------------------------------
      const std::string & AppName(const std::string & appName);

      //----------------------------------------------------------------------
      //!  Returns the parameters that will be appended to the URI when
      //!  making queries.  Used for things like X-Plex-Token when talking
      //!  to Plex.
      //----------------------------------------------------------------------
      const std::vector<std::string> & Params() const;

      //----------------------------------------------------------------------
      //!  Sets the parameters that will be appended to the URI when
      //!  making queries.  Used for things like X-Plex-Token when talking
      //!  to Plex.  Each param is of the form 'name=value'.
      //----------------------------------------------------------------------
      const std::vector<std::string> &
      Params(const std::vector<std::string> & params);

      //----------------------------------------------------------------------
      //!  For applications which emit XML, the path we'll use to determine
      //!  if an XML response is satisfactory.
      //----------------------------------------------------------------------
      const std::string & Xpath() const;

      //----------------------------------------------------------------------
      //!  For applications which emit XML, the path we'll use to determine
      //!  if an XML response is satisfactory.
      //----------------------------------------------------------------------
      const std::string & Xpath(const std::string & xpath);

      //----------------------------------------------------------------------
      //!  Returns @c true if the target's cerficate should be validated, else
      //!  returns @c false.
      //----------------------------------------------------------------------
      bool ValidateCertificate() const;
      
      //----------------------------------------------------------------------
      //!  If @c validateCertificate is @c true, we will validate the target's
      //!  certificate when testing the target's availability.  Else we will
      //!  ignore the validity of the target's certificate.
      //----------------------------------------------------------------------
      bool ValidateCertificate(bool validateCertificate);

      //----------------------------------------------------------------------
      //!  Returns a JSON representation of the target.
      //----------------------------------------------------------------------
      nlohmann::json ToJson() const;

      //----------------------------------------------------------------------
      //!  Populates the target from the information in the JSON object @c j.
      //!  Returns true on success, false on failure.
      //----------------------------------------------------------------------
      bool FromJson(const nlohmann::json & j);
      
    private:
      std::string               _uri;
      bool                      _validateCertificate;
      std::string               _xpath;
      std::vector<std::string>  _params;
      std::string               _appName;
    };
    
  }  // namespace Mcrover

}  // namespace Dwm

#endif  // _DWMMCROVERWEBAPPTARGET_HH_
