//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/classes/include/DwmMcroverAlertBowl.hh 12026 $
// @(#) $Id: DwmMcroverAlertBowl.hh 12026 2022-06-07 05:18:54Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverAlertBowl.hh
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::AlertBowl class declaration
//---------------------------------------------------------------------------

#ifndef _DWMMCROVERALERTBOWL_HH_
#define _DWMMCROVERALERTBOWL_HH_

#include <shared_mutex>

#include "DwmDescriptorIOCapable.hh"
#include "DwmFileIOCapable.hh"
#include "DwmStreamIOCapable.hh"
#include "DwmStreamedLengthCapable.hh"
#include "DwmMcroverAlertOrigin.hh"
#include "DwmMcroverAlertVariants.hh"

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  Encapsulates a map of timestamps keyed by alert origin.  We need
    //!  this in order to group the same alert reported by multiple origins.
    //!  For example, we might have mcroverd running on multiple hosts and
    //!  testing the same remote service.
    //------------------------------------------------------------------------
    class AlertBowlValue
      : public DescriptorIOCapable, public FileIOCapable,
        public StreamIOCapable, public StreamedLengthCapable
    {
    public:
      //----------------------------------------------------------------------
      //!  Default constructor
      //----------------------------------------------------------------------
      AlertBowlValue() = default;
      
      //----------------------------------------------------------------------
      //!  Copy constructor
      //----------------------------------------------------------------------
      AlertBowlValue(const AlertBowlValue & abv);

      //----------------------------------------------------------------------
      //!  Move constructor
      //----------------------------------------------------------------------
      AlertBowlValue(AlertBowlValue && abv)
          : _mtx(), _data(std::move(abv._data))
      {
      }

      //----------------------------------------------------------------------
      //!  Assignment operator
      //----------------------------------------------------------------------
      AlertBowlValue & operator = (const AlertBowlValue & abv);
      
      bool Add(const AlertOrigin & origin, uint64_t t);
      bool Add(const AlertBowlValue & abv);
      bool Remove(const AlertOrigin & origin);
      bool RemovePack(const std::string & packName);
      bool Refresh(const AlertBowlValue & abv);
      bool Clear();
      bool Empty() const;
      bool operator == (const AlertBowlValue & abv) const;
      bool GetConcise(time_t & t, std::string & origins) const;
      std::string DisplayString() const;
      bool GetByPack(const std::string & packName, AlertBowlValue & abv) const;
      bool GetByOrigin(const AlertOrigin & origin, AlertBowlValue & abv) const;
      std::pair<AlertOrigin,uint64_t> Oldest() const;
      std::pair<AlertOrigin,uint64_t> Newest() const;
      
      friend std::ostream &
      operator << (std::ostream & os, const AlertBowlValue & abv);
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      std::istream & Read(std::istream & is) override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ssize_t Read(int fd) override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      size_t Read(FILE * f) override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      std::ostream & Write(std::ostream & os) const override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ssize_t Write(int fd) const override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      size_t Write(FILE *f) const override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      uint64_t StreamedLength() const override;
      
    private:
      mutable std::shared_mutex       _mtx;
      std::map<AlertOrigin,uint64_t>  _data;
    };

    //------------------------------------------------------------------------
    //!  Encapsulates a collection of alerts.  This is the main container
    //!  used by mcroverd to keep track of all active alerts.
    //------------------------------------------------------------------------
    class AlertBowl
      : public DescriptorIOCapable, public FileIOCapable,
        public StreamIOCapable, public StreamedLengthCapable
    {
    public:
      typedef std::variant<DWM_MCROVER_ALERT_VARIANTS>    KeyType;
      typedef std::pair<KeyType,AlertBowlValue>           EntryType;
        
      bool Load(const std::string & fileName);
      bool Save(const std::string & fileName) const;
      bool Add(const AlertOrigin & origin, const KeyType & key,
               uint64_t t = time((time_t *)0));
      bool Add(const AlertBowl & alerts);
      bool Remove(const AlertOrigin & origin, const KeyType & key);
      bool RemoveOrigin(const AlertOrigin & origin);
      bool RemovePack(const std::string & packName);
      void Refresh(const AlertBowl & alerts);
      bool Empty() const;
      size_t Size() const;
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      void Clear();

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      bool
      GetConcise(std::vector<std::pair<time_t,std::string>> & alerts) const;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      void Get(std::vector<std::pair<KeyType,AlertBowlValue>> & alerts) const;
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      bool GetByPack(const std::string & packName,
                     AlertBowl & alertBowl) const;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      bool GetByOrigin(const AlertOrigin & origin,
                       AlertBowl & alertBowl) const;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      friend std::ostream &
      operator << (std::ostream & os, const AlertBowl & ab);
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      std::istream & Read(std::istream & is) override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ssize_t Read(int fd) override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      size_t Read(FILE * f) override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      std::ostream & Write(std::ostream & os) const override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ssize_t Write(int fd) const override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      size_t Write(FILE *f) const override;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      uint64_t StreamedLength() const override;

      //----------------------------------------------------------------------
      //!  This really only exists for unit tests at the moment.
      //----------------------------------------------------------------------
      bool operator == (const AlertBowl & ab) const;
        
    private:
      mutable std::shared_mutex         _mtx;
      std::map<KeyType,AlertBowlValue>  _data;
    };
    
  }  // namespace Mcrover

}  // namespace Dwm

#endif  // _DWMMCROVERALERTBOWL_HH_
