//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.1/apps/mcrover/mcrover.cc 12027 $
// @(#) $Id: mcrover.cc 12027 2022-06-07 15:52:05Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file mchmc.cc
//!  \author Daniel W. McRobb
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <locale.h>
  #include <sys/ioctl.h>
}

#include <atomic>
#include <chrono>
#include <iostream>
#include <regex>
#include <sstream>
#include <thread>

#include "DwmDateTime.hh"
#include "DwmSignal.hh"
#include "DwmSocket.hh"
#include "DwmSysLogger.hh"
#include "DwmCredencePeer.hh"
#include "DwmMcroverAlertBowl.hh"
#include "DwmMcroverPackMember.hh"
#include "DwmMcroverConfig.hh"
#include "DwmMcroverUtils.hh"
#include "DwmMcroverAlertsWindow.hh"
#include "DwmMcroverStatusWindow.hh"
#include "DwmMcroverTitleWindow.hh"

extern "C" {
  #include <curses.h>
}

using namespace std;
using namespace Dwm;
using Mcrover::AlertBowl, Mcrover::AlertBowlValue;

static atomic<bool>  g_needResize = false;
static atomic<bool>  g_shouldExit = false;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void SigWinchHandler(int sigNum)
{
  if (SIGWINCH == sigNum) {
    g_needResize = true;
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void SigHupHandler(int sigNum)
{
  if (SIGHUP == sigNum) {
    g_shouldExit = true;
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void SigTermHandler(int sigNum)
{
  if (SIGTERM == sigNum) {
    g_shouldExit = true;
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void
SortByTime(vector<pair<AlertBowl::KeyType,AlertBowlValue>> & alerts)
{
  sort (alerts.begin(), alerts.end(),
        [] (const auto & alert1, const auto & alert2)
        { return (alert1.second.Newest().second
                  > alert2.second.Newest().second); });
  return;
}
        
//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void ResizeWindows(Mcrover::TitleWindow & titleWin,
                   Mcrover::AlertsWindow & alertsWin,
                   Mcrover::StatusWindow & statusWin)
{
  struct winsize   size;
  ioctl(fileno(stdout), TIOCGWINSZ, (char*) &size);
  resizeterm(size.ws_row, size.ws_col);
  getmaxyx(stdscr, LINES, COLS);
  titleWin.Resize();
  alertsWin.Resize();
  statusWin.Resize();
  g_needResize = false;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool GetMcroverHosts(const Mcrover::Config & config,
                            vector<Mcrover::PackMemberConfig> & hosts)
{
  namespace baip = boost::asio::ip;
  hosts.clear();
  if (config.RunService()) {
    //  Local host is running service.  Try there first.
    Mcrover::PackMemberConfig  localAsPeer;
    localAsPeer.Name(Mcrover::Utils::ThisHostName());
    localAsPeer.AddAddress(baip::tcp::endpoint(baip::address::from_string("127.0.0.1"), 2123));
    localAsPeer.AddAddress(baip::tcp::endpoint(baip::address::from_string("::1"), 2123));
    hosts.push_back(localAsPeer);
  }
  for (const auto & memberConfig : config.MyPack().Members()) {
    hosts.push_back(memberConfig.second);
  }
  return !(hosts.empty());
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool GetAlerts(const Mcrover::PackMemberConfig & host,
                      AlertBowl & alerts,
                      Mcrover::RoverAlert::FailureType & failure)
{
  bool  rc = false;
  Mcrover::PackMember  packMember(host);
  return packMember.GetAllAlerts(alerts, failure);
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool
GetAlerts(const Mcrover::Config & config,
          std::string & chosenHost,
          vector<pair<AlertBowl::KeyType,AlertBowlValue>> & alerts)
{
  bool  rc = false;
  chosenHost.clear();
  vector<Mcrover::PackMemberConfig>  hosts;
  if (GetMcroverHosts(config, hosts)) {
    AlertBowl  alertBowl;
    for (const auto & host : hosts) {
      Mcrover::RoverAlert::FailureType  failure;
      if (GetAlerts(host, alertBowl, failure)) {
        alertBowl.Get(alerts);
        chosenHost = host.Name();
        rc = true;
        break;
      }
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  string  cfgPath("/usr/local/etc/mcroverd.cfg");
  if (argc >= 2) {
    cfgPath = argv[argc-1];
  }

  Mcrover::Config  config;
  if (! config.Parse(cfgPath)) {
    cerr << "Bad configuration " << cfgPath << '\n';
    return 1;
  }
  setlocale(LC_ALL, "");
  initscr();
  noecho();
  nodelay(stdscr, true);
  keypad(stdscr, true);
  halfdelay(2);
  
  start_color();
  curs_set(0);
  
  Mcrover::TitleWindow   titleWindow;
  Mcrover::AlertsWindow  alertsWindow;
  Mcrover::StatusWindow  statusWindow("");
  ResizeWindows(titleWindow, alertsWindow, statusWindow);
  
  Dwm::SysLogger::Open("mcrover", LOG_PID, LOG_LOCAL0);
  Syslog(LOG_INFO, "Started");
  Syslog(LOG_INFO, "COLORS: %d COLOR_PAIRS: %d", COLORS, COLOR_PAIRS);
  
  Dwm::Signal sigWinch(SIGWINCH);
  sigWinch.PushHandler(SigWinchHandler);
  Dwm::Signal sigHup(SIGHUP);
  sigHup.PushHandler(SigHupHandler);
  Dwm::Signal sigTerm(SIGTERM);
  sigTerm.PushHandler(SigTermHandler);
  
  vector<pair<AlertBowl::KeyType,AlertBowlValue>> alerts;
  
  time_t  nextPollTime = 0;
  string  chosenHost;
  
  for (;;) {
    if (g_shouldExit) {
      endwin();
      exit(0);
    }
    time_t  now = time((time_t *)0);
    if (now > nextPollTime) {
      nextPollTime = now + 60;
      if (GetAlerts(config, chosenHost, alerts)) {
        SortByTime(alerts);
        titleWindow.Update(alerts.size());
        alertsWindow.Update(alerts);
        statusWindow.Update(chosenHost, alerts.size());
        wrefresh(stdscr);
      }
      else {
        Syslog(LOG_ERR, "Failed to get alerts");
      }
    }
    if (g_needResize) {
      ResizeWindows(titleWindow, alertsWindow, statusWindow);
      titleWindow.Update(alerts.size());
      alertsWindow.Update(alerts);
      statusWindow.Update(chosenHost, alerts.size());
    }
    int  userChar = getch();
    if (ERR != userChar) {
      switch (userChar) {
        case KEY_DOWN:
          alertsWindow.Scroll(1);
          alertsWindow.Update(alerts);
          break;
        case 'n':
        case KEY_NPAGE:
          alertsWindow.ScrollPage(1);
          alertsWindow.Update(alerts);
          break;
        case KEY_UP:
          alertsWindow.Scroll(-1);
          alertsWindow.Update(alerts);
          break;
        case 'p':
        case KEY_PPAGE:
          alertsWindow.ScrollPage(-1);
          alertsWindow.Update(alerts);
          break;
        case 'q':
          endwin();
          exit(0);
          break;
        default:
          titleWindow.Update(alerts.size());
          alertsWindow.Update(alerts);
          statusWindow.Update(chosenHost, alerts.size());
          break;
      }
    }
    else {
      this_thread::sleep_for(chrono::milliseconds(200));
    }
  }
}
