//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverUPSAlert.cc 11886 $
// @(#) $Id: DwmMcroverUPSAlert.cc 11886 2021-04-07 05:16:39Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverUPSAlert.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::UPSAlert class implementation
//---------------------------------------------------------------------------

#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmIOUtils.hh"
#include "DwmStreamIO.hh"
#include "DwmSvnTag.hh"
#include "DwmMcroverUPSAlert.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverUPSAlert.cc 11886 $");

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    UPSAlert::UPSAlert(const std::string & name,
                       const Ipv4Address & addr,
                       const std::string & upsName,
                       const UPSStatus & status)
        : _name(name), _addr(addr), _upsName(upsName), _status(status)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & UPSAlert::Name() const
    {
      return _name;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & UPSAlert::Name(const std::string & name)
    {
      _name = name;
      return _name;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & UPSAlert::Address() const
    {
      return _addr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & UPSAlert::Address(const Ipv4Address & addr)
    {
      _addr = addr;
      return _addr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string UPSAlert::UPSName() const
    {
      return _upsName;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string UPSAlert::UPSName(const std::string & upsName)
    {
      _upsName = upsName;
      return _upsName;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const UPSStatus & UPSAlert::Status() const
    {
      return _status;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const UPSStatus & UPSAlert::Status(const UPSStatus & status)
    {
      _status = status;
      return _status;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool UPSAlert::operator == (const UPSAlert & alert) const
    {
      return ((_name == alert._name)
              && (_addr == alert._addr)
              && (_upsName == alert._upsName)
              && (_status == alert._status));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool UPSAlert::operator < (const UPSAlert & alert) const
    {
      bool  rc = false;
      if (_name < alert._name) { rc = true; }
      else if (_name == alert._name) {
        if (_addr < alert._addr) { rc = true; }
        else if (_addr == alert._addr) {
          if (_upsName < alert._upsName) { rc = true; }
          else if (_upsName == alert._upsName) {
            if (_status < alert._status) { rc = true; }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t UPSAlert::Priority() const
    {
      uint8_t  priority = 2;
      if (_status.Status() == UPSStatus::e_upsLowBattery) {
        priority = 1;
      }
      return priority;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json UPSAlert::ToJson() const
    {
      nlohmann::json  j;
      j["name"] = _name;
      j["addr"] = (std::string)_addr;
      j["ups"] = _upsName;
      j["status"] = _status.Status();
      return j;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool UPSAlert::FromJson(const nlohmann::json & j)
    {
      bool  rc = false;
      if (j.is_object()) {
        auto  name = j.find("name");
        if ((name != j.end()) && name->is_string()) {
          _name = name->get<std::string>();
          auto  addr = j.find("addr");
          if ((addr != j.end()) && addr->is_string()) {
            _addr = Ipv4Address(addr->get<std::string>());
            auto  ups = j.find("ups");
            if ((ups != j.end()) && ups->is_string()) {
              _upsName = ups->get<std::string>();
              auto  status = j.find("status");
              if ((status != j.end()) && status->is_number()) {
                _status = status->get<decltype(_status.Status())>();
                rc = true;
              }
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string UPSAlert::DisplayString() const
    {
      std::string  rc;
      if (! _name.empty()) {
        rc += _name + ':';
      }
      else {
        rc += (std::string)_addr + ':';
      }
      rc += _upsName + ':';
      rc += _status.StatusName() + " UPS";
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & UPSAlert::Read(std::istream & is)
    {
      if (is) {
        if (Dwm::StreamIO::Read(is, _name)) {
          if (_addr.Read(is)) {
            if (Dwm::StreamIO::Read(is, _upsName)) {
              uint32_t status;
              if (Dwm::StreamIO::Read(is, status)) {
                _status = UPSStatus((UPSStatus::StatusEnum)status);
              }
            }
          }
        }
      }
      return is;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t UPSAlert::Read(int fd)
    {
      ssize_t  rc = -1;
      if (0 <= fd) {
        ssize_t  bytesRead = Dwm::DescriptorIO::Read(fd, _name);
        if (bytesRead > 0) {
          rc = bytesRead;
          bytesRead = _addr.Read(fd);
          if (bytesRead > 0) {
            rc += bytesRead;
            bytesRead = Dwm::DescriptorIO::Read(fd, _upsName);
            if (bytesRead > 0) {
              rc += bytesRead;
              uint32_t  status;
              bytesRead = Dwm::DescriptorIO::Read(fd, status);
              if (sizeof(status) == bytesRead) {
                rc += bytesRead;
                _status	= UPSStatus((UPSStatus::StatusEnum)status);
              }
              else {
                rc = -1;
              }
            }
            else {
              rc = -1;
            }
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t UPSAlert::Read(FILE * f)
    {
      size_t  rc = 0;
      if (f) {
        if (Dwm::FileIO::Read(f, _name)) {
          if (_addr.Read(f)) {
            if (Dwm::FileIO::Read(f, _upsName)) {
              uint32_t  status;
              if (Dwm::FileIO::Read(f, status)) {
                _status	= UPSStatus((UPSStatus::StatusEnum)status);
                rc = 1;
              }
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & UPSAlert::Write(std::ostream & os) const
    {
      if (os) {
        if (Dwm::StreamIO::Write(os, _name)) {
          if (_addr.Write(os)) {
            if (Dwm::StreamIO::Write(os, _upsName)) {
              uint32_t  status = _status.Status();
              Dwm::StreamIO::Write(os, status);
            }
          }
        }
      }
      return os;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t UPSAlert::Write(int fd) const
    {
      ssize_t rc = -1;
      if (0 <= fd) {
        ssize_t  bytesWritten = Dwm::DescriptorIO::Write(fd, _name);
        if (bytesWritten > 0) {
          rc = bytesWritten;
          bytesWritten = _addr.Write(fd);
          if (_addr.StreamedLength() == bytesWritten) {
            rc += bytesWritten;
            bytesWritten = Dwm::DescriptorIO::Write(fd, _upsName);
            if (bytesWritten > 0) {
              rc += bytesWritten;
              uint32_t  status = _status.Status();
              bytesWritten = Dwm::DescriptorIO::Write(fd, status);
              if (sizeof(status) == bytesWritten) {
                rc += bytesWritten;
              }
              else {
                rc = -1;
              }
            }
            else {
              rc = -1;
            }
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
            
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t UPSAlert::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (f) {
        if (Dwm::FileIO::Write(f, _name)) {
          if (_addr.Write(f)) {
            if (Dwm::FileIO::Write(f, _upsName)) {
              uint32_t  status = _status.Status();
              rc = Dwm::FileIO::Write(f, status);
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t UPSAlert::StreamedLength() const
    {
      return (Dwm::IOUtils::StreamedLength(_name) + _addr.StreamedLength()
              + Dwm::IOUtils::StreamedLength(_upsName) + sizeof(uint32_t));
    }
    
  }  // namespace Mcrover
  
}  // namespace Dwm
