//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020, 2022, 2023
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverTargetHostConfig.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::TargetHostConfig class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
  #include <netdb.h>
}

#include <iomanip>

#include "DwmDnsUtils.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverTargetHostConfig.hh"
#include "DwmMcroverUtils.hh"

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    TargetHostConfig::TargetHostConfig()
        : _name(), _address(INADDR_NONE), _address6(), _tcp4Ports(),
          _tcp6Ports(), _tcp4PortsDenied(), _tcp6PortsDenied(),
          _expectedPrograms(), _expected6Programs(), _smtpPorts(),
          _smtp6Ports(), _ups(), _webTargets(), _dnsResolve()
    {}

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & TargetHostConfig::Name() const
    {
      return _name;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & TargetHostConfig::Name(const std::string & name)
    {
      _name = name;
      return _name;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & TargetHostConfig::Address() const
    {
      return _address;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & TargetHostConfig::Address(const Ipv4Address & addr)
    {
      _address = addr;
      return _address;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv6Address & TargetHostConfig::Address6() const
    {
      return _address6;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv6Address & TargetHostConfig::Address6(const Ipv6Address & addr)
    {
      _address6 = addr;
      return _address6;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Ipv4Address TargetHostConfig::PreferredAddress() const
    {
      Ipv4Address  rc = _address;
      if (INADDR_NONE == rc.Raw()) {
        rc = Utils::GetHostIpv4Addr(_name);
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<uint16_t> & TargetHostConfig::TCP4Ports() const
    {
      return _tcp4Ports;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<uint16_t> &
    TargetHostConfig::TCP4Ports(const vector<uint16_t> & ports)
    {
      _tcp4Ports = ports;
      return _tcp4Ports;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<uint16_t> & TargetHostConfig::TCP6Ports() const
    {
      return _tcp6Ports;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<uint16_t> &
    TargetHostConfig::TCP6Ports(const vector<uint16_t> & ports)
    {
      _tcp6Ports = ports;
      return _tcp6Ports;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<uint16_t> & TargetHostConfig::TCP4PortsDenied() const
    {
      return _tcp4PortsDenied;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<uint16_t> &
    TargetHostConfig::TCP4PortsDenied(const vector<uint16_t> & ports)
    {
      _tcp4PortsDenied = ports;
      return _tcp4PortsDenied;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<uint16_t> & TargetHostConfig::TCP6PortsDenied() const
    {
      return _tcp6PortsDenied;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<uint16_t> &
    TargetHostConfig::TCP6PortsDenied(const vector<uint16_t> & ports)
    {
      _tcp6PortsDenied = ports;
      return _tcp6PortsDenied;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const set<RPCProgramId> & TargetHostConfig::ExpectedPrograms() const
    {
      return _expectedPrograms;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const set<RPCProgramId> &
    TargetHostConfig::ExpectedPrograms(const vector<RPCProgramId> & progs)
    {
      _expectedPrograms.clear();
      for (const auto & prog : progs) {
        _expectedPrograms.insert(prog);
      }
      return _expectedPrograms;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const set<RPCProgramId> & TargetHostConfig::Expected6Programs() const
    {
      return _expected6Programs;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const set<RPCProgramId> &
    TargetHostConfig::Expected6Programs(const vector<RPCProgramId> & progs)
    {
      _expected6Programs.clear();
      for (const auto & prog : progs) {
        _expected6Programs.insert(prog);
      }
      return _expected6Programs;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<uint16_t> & TargetHostConfig::SMTPPorts() const
    {
      return _smtpPorts;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::vector<uint16_t> &
    TargetHostConfig::SMTPPorts(const std::vector<uint16_t> & ports)
    {
      _smtpPorts = ports;
      return _smtpPorts;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<uint16_t> & TargetHostConfig::SMTP6Ports() const
    {
      return _smtp6Ports;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::vector<uint16_t> &
    TargetHostConfig::SMTP6Ports(const std::vector<uint16_t> & ports)
    {
      _smtp6Ports = ports;
      return _smtp6Ports;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<string> & TargetHostConfig::UPS() const
    {
      return _ups;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<string> & TargetHostConfig::UPS(const vector<string> & upses)
    {
      _ups = upses;
      return _ups;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::vector<WebTarget> & TargetHostConfig::WebTargets() const
    {
      return _webTargets;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::vector<WebTarget> &
    TargetHostConfig::WebTargets(const std::vector<WebTarget> & targets)
    {
      _webTargets = targets;
      return _webTargets;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::vector<WebAppTarget> & TargetHostConfig::WebAppTargets() const
    {
      return _webAppTargets;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::vector<WebAppTarget> &
    TargetHostConfig::WebAppTargets(const std::vector<WebAppTarget> & targets)
    {
      return _webAppTargets = targets;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<Dns::MessageQuestion> & TargetHostConfig::DNSResolve() const
    {
      return _dnsResolve;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<Dns::MessageQuestion> &
    TargetHostConfig::DNSResolve(const vector<Dns::MessageQuestion> & resolveList)
    {
      _dnsResolve = resolveList;
      return _dnsResolve;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<TcpDstAddr> & TargetHostConfig::CredencePeerTargets() const
    {
      return _credencePeerTargets;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const vector<TcpDstAddr> &
    TargetHostConfig::CredencePeerTargets(const vector<TcpDstAddr> & targets)
    {
      return _credencePeerTargets = targets;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void PrintTcpPorts(std::ostream & os, const std::string & name,
                              const std::vector<uint16_t> & ports)
    {
      if (! ports.empty()) {
        os << "        " << name << " = [ ";
        int   indent = 13 + name.size();
        int   col = indent;
        auto  iter = ports.begin();
        setservent(1);
        struct servent  *servEntry = getservbyport(htons(*iter), "tcp");
        if (servEntry) {
          os << servEntry->s_name;
          col += strlen(servEntry->s_name);
        }
        else {
          os << *iter;
          col += to_string(*iter).size();
        }
        ++iter;
        for ( ; iter != ports.end(); ++iter) {
          servEntry = getservbyport(htons(*iter), "tcp");
          os << ", ";
          col += 2;
          if (servEntry) {
            col += strlen(servEntry->s_name);
            if (col > 76) {
              os << '\n' << setw(indent) << ' ';
              col = indent;
            }
            os << servEntry->s_name;
          }
          else {
            col += 2 + to_string(*iter).size();
            if (col > 76) {
              os << '\n' << setw(indent) << ' ';
              col = indent;
            }
            os  << *iter;
          }
        }
        endservent();
        os << " ];\n";
      }
      
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PrintRPCPrograms(std::ostream & os, const std::set<RPCProgramId> & progs)
    {
      if (! progs.empty()) {
        os << "        RPC = [\n";
        auto  iter = progs.begin();
        struct protoent  *protoEntry = getprotobynumber(iter->Protocol());
        if (protoEntry) {
          os << "            { name = \"" << iter->Name() << "\";"
             << " protocol = \"" << protoEntry->p_name << "\";"
             << " version = " << iter->Version() << "; }";
        }
        ++iter;
        for ( ; iter != progs.end(); ++iter) {
          protoEntry = getprotobynumber(iter->Protocol());
          if (protoEntry) {
            os << ",\n            { name = \"" << iter->Name() << "\";"
               << " protocol = \"" << protoEntry->p_name << "\";"
               << " version = " << iter->Version() << "; }";
          }
        }
        os << "\n        ];\n";
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void PrintUPS(std::ostream & os,
                         const std::vector<std::string> & ups)
    {
      if (! ups.empty()) {
        os << "        UPS = [ ";
        auto  iter = ups.begin();
        os << '"' << *iter << '"';
        ++iter;
        for ( ; iter != ups.end(); ++iter) {
          os << ", \"" << *iter << '"';
        }
        os << " ];\n";
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void PrintWebTarget(std::ostream & os, const WebTarget & webTarget)
    {
      os << "            { uri = \"" << webTarget.URI() << '"';
      if (! webTarget.GoodResponses().Pairs().empty()) {
        os << "; status = [ ";
        auto  iter = webTarget.GoodResponses().Pairs().begin();
        os << iter->first;
        if (iter->second != iter->first) {
          os << '-' << iter->second;
        }
        ++iter;
        for ( ; iter != webTarget.GoodResponses().Pairs().end(); ++iter) {
          os << ", " << iter->first;
          if (iter->second != iter->first) {
            os << '-' << iter->second;
          }
        }
        os << " ]";
      }
      os << "; }";
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void PrintWebTargets(std::ostream & os,
                                const std::vector<WebTarget> & webTargets)
    {
      if (! webTargets.empty()) {
        os << "        WEB = [\n";
        auto  iter = webTargets.begin();
        PrintWebTarget(os, *iter);
        ++iter;
        for ( ; iter != webTargets.end(); ++iter) {
          os << ",\n";
          PrintWebTarget(os, *iter);
        }
        os << "\n        ];\n";
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void PrintWebAppTarget(std::ostream & os,
                                  const WebAppTarget & appTarget)
    {
      os << "            { uri = \"" << appTarget.URI() << "\";\n"
         << "              appname = \"" << appTarget.AppName() << "\";\n";
      if (! appTarget.Params().empty()) {
        os << "              params = [\n";
        auto pit = appTarget.Params().begin();
        os << "                \"" << *pit << "\"";
        ++pit;
        for ( ; pit != appTarget.Params().end(); ++pit) {
          os << ",\n                \"" << *pit << "\"";
        }
        os << "\n              ];\n";
      }
      if (! appTarget.Xpath().empty()) {
        os << "              xpath = \"" << appTarget.Xpath() << "\";\n";
      }
      os << "            }";
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PrintWebAppTargets(std::ostream & os,
                       const std::vector<WebAppTarget> & appTargets)
    {
      if (! appTargets.empty()) {
        os << "        WEBAPP = [\n";
        auto  iter = appTargets.begin();
        PrintWebAppTarget(os, *iter);
        ++iter;
        for ( ; iter != appTargets.end(); ++iter) {
          os << ",\n";
          PrintWebAppTarget(os, *iter);
        }
        os << "\n        ];\n";
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PrintCredencePeerTarget(ostream & os, const TcpDstAddr & target)
    {
      os << "            { name = \"" << target.Name() << "\";"
         << " address = \"" << target.Address() << "\";"
         << " port = " << target.Port() << ';'
         << " }";
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void
    PrintCredencePeerTargets(ostream & os, const vector<TcpDstAddr> & targets)
    {
      if (! targets.empty()) {
        os << "        CREDENCE = [\n";
        auto  iter = targets.begin();
        PrintCredencePeerTarget(os, *iter);
        ++iter;
        for ( ; iter != targets.end(); ++iter) {
          os << ",\n";
          PrintCredencePeerTarget(os, *iter);
        }
        os << "\n        ];\n";
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static void PrintDNS(std::ostream & os,
                         const std::vector<Dns::MessageQuestion> & questions)
    {
      if (! questions.empty()) {
        os << setw(8) << ' ' << "DNS = {\n"
           << setw(12) << ' ' << "#------------------------------------------"
          "----------------------\n"
           << setw(12) << ' ' << "#  Currently supported qtypes: A, NS,"
          " CNAME, SOA, PTR, MX, AAAA,\n"
           << setw(12) << ' ' << "#  LOC.  Note that if no qtype is"
          " specified, we'll assume PTR for\n"
           << setw(12) << ' ' << "#  IP addresses and A for anything else.\n"
           << setw(12) << ' ' << "#------------------------------------------"
          "----------------------\n"
           << setw(12) << ' ' << "resolve = [\n";
        auto  iter = questions.begin();
        os << setw(16) << ' ' << "{ qtype = \"" << iter->QTypeString()
           << "\"; qname = \"";
        if (iter->QType() == Dns::MessageQuestion::k_typePTR) {
          string  fromArpa;
          Dns::FromArpa(iter->QName(), fromArpa);
          os << fromArpa;
        }
        else {
          os << iter->QName();
        }
        os << "\"; }";
        ++iter;
        for ( ; iter != questions.end(); ++iter) {
          os << ",\n" << setw(16) << ' ' << "{ qtype = \""
             << iter->QTypeString() << "\"; qname = \"";
          if (iter->QType() == Dns::MessageQuestion::k_typePTR) {
            string  fromArpa;
            Dns::FromArpa(iter->QName(), fromArpa);
            os << fromArpa;
          }
          else {
            os << iter->QName();
          }
          os << "\"; }";
        }
        os << '\n' << setw(12) << ' ' << "];\n"
           << setw(8) << ' ' << "};\n";
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream &
    operator << (std::ostream & os, const TargetHostConfig & cfg)
    {
      os << "    \"" << cfg._name << "\" {\n";
      if (cfg._address != Ipv4Address()) {
        os << "        ipv4 = " << cfg._address << ";\n";
      }
      if (cfg._address6 != Ipv6Address()) {
        os << "        ipv6 = " << cfg._address6 << ";\n";
      }
      PrintTcpPorts(os, "tcp4", cfg._tcp4Ports);
      PrintTcpPorts(os, "tcp6", cfg._tcp6Ports);
      PrintTcpPorts(os, "tcp4Denied", cfg._tcp4PortsDenied);
      PrintTcpPorts(os, "tcp6Denied", cfg._tcp6PortsDenied);
      PrintRPCPrograms(os, cfg._expectedPrograms);
      PrintTcpPorts(os, "SMTP", cfg._smtpPorts);
      PrintTcpPorts(os, "SMTP6", cfg._smtp6Ports);
      PrintUPS(os, cfg._ups);
      PrintWebTargets(os, cfg._webTargets);
      PrintWebAppTargets(os, cfg._webAppTargets);
      PrintCredencePeerTargets(os, cfg._credencePeerTargets);
      PrintDNS(os, cfg._dnsResolve);
      
      os << "    };\n";
      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json TargetHostConfig::ToJson() const
    {
      nlohmann::json  j;
      j["name"] = _name;
      if (_address.Raw() != 0) {
        j["address"] = (string)_address;
      }
      if (_address6 != Ipv6Address("::")) {
        j["address6"] = (string)_address6;
      }
      if (! _tcp4Ports.empty()) {
        for (size_t i = 0; i < _tcp4Ports.size(); ++i) {
          j["tcp4Ports"][i] = _tcp4Ports[i];
        }
      }
      if (! _tcp6Ports.empty()) {
        for (size_t i = 0; i < _tcp6Ports.size(); ++i) {
          j["tcp6Ports"][i] = _tcp6Ports[i];
        }
      }
      if (! _tcp4PortsDenied.empty()) {
        for (size_t i = 0; i < _tcp4PortsDenied.size(); ++i) {
          j["tcp4PortsDenied"][i] = _tcp4PortsDenied[i];
        }
      }
      if (! _tcp6PortsDenied.empty()) {
        for (size_t i = 0; i < _tcp6PortsDenied.size(); ++i) {
          j["tcp6PortsDenied"][i] = _tcp6PortsDenied[i];
        }
      }
      if (! _expectedPrograms.empty()) {
        size_t  i = 0;
        for (auto it = _expectedPrograms.begin();
             it != _expectedPrograms.end(); ++it) {
          j["rpcPrograms"][i] = it->ToJson();
          ++i;
        }
      }
      if (! _expected6Programs.empty()) {
        size_t  i = 0;
        for (auto it = _expected6Programs.begin();
             it != _expected6Programs.end(); ++it) {
          j["rpc6Programs"][i] = it->ToJson();
          ++i;
        }
      }
      if (! _smtpPorts.empty()) {
        for (size_t i = 0; i < _smtpPorts.size(); ++i) {
          j["smtpPorts"][i] = _smtpPorts[i];
        }
      }
      if (! _smtp6Ports.empty()) {
        for (size_t i = 0; i < _smtp6Ports.size(); ++i) {
          j["smtp6Ports"][i] = _smtp6Ports[i];
        }
      }
      if (! _ups.empty()) {
        for (size_t i = 0; i < _ups.size(); ++i) {
          j["ups"][i] = _ups[i];
        }
      }
      if (! _webTargets.empty()) {
        for (size_t i = 0; i < _webTargets.size(); ++i) {
          j["webTargets"][i] = _webTargets[i].ToJson();
        }
      }
      if (! _webAppTargets.empty()) {
        for (size_t i = 0; i < _webAppTargets.size(); ++i) {
          j["webAppTargets"][i] = _webAppTargets[i].ToJson();
        }
      }
      if (! _dnsResolve.empty()) {
        for (size_t i = 0; i < _dnsResolve.size(); ++i) {
          nlohmann::json  dnsqj;
          dnsqj["qname"] = _dnsResolve[i].QName();
          dnsqj["qtype"] = _dnsResolve[i].QType();
          j["dnsResolve"][i] = dnsqj;
        }
      }
      if (! _credencePeerTargets.empty()) {
        for (size_t i = 0; i < _credencePeerTargets.size(); ++i) {
          j["credencePeerTargets"][i] = _credencePeerTargets[i].ToJson();
        }
      }
      return j;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename T>
    static void FromJsonArray(const nlohmann::json & j, vector<T> & v)
    {
      for (size_t i = 0; i < j.size(); ++i) {
        v.push_back(j[i].get<T>());
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename T>
    static bool FromJsonArray(const nlohmann::json & j, const string & member,
                              vector<T> & v)
    {
      bool  rc = false;
      auto it = j.find(member);
      if ((it != j.end()) && it->is_array()) {
        FromJsonArray(*it, v);
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool FromJsonArray(const nlohmann::json & j, const string & member,
                              std::set<RPCProgramId> & rpcids)
    {
      bool  rc = false;
      auto it = j.find(member);
      if ((it != j.end()) && it->is_array()) {
        size_t  i = 0;
        for ( ; i < it->size(); ++i) {
          RPCProgramId  rpcid;
          if (rpcid.FromJson((*it)[i])) {
            rpcids.insert(rpcid);
          }
          else {
            break;
          }
        }
        rc = (it->size() == i);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename T>
    static bool ObjsFromJsonArray(const nlohmann::json & j,
                                  const string & member,
                                  std::vector<T> & objs)
    {
      bool  rc = false;
      auto it = j.find(member);
      if ((it != j.end()) && it->is_array()) {
        size_t  i = 0;
        for ( ; i < it->size(); ++i) {
          T  obj;
          if (obj.FromJson((*it)[i])) {
            objs.push_back(obj);
          }
        }
        rc = (it->size() == i);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool ObjsFromJsonArray(const nlohmann::json & j,
                                  const string & member,
                                  vector<Dns::MessageQuestion> & dnsqs)
    {
      bool  rc = false;
      auto it = j.find(member);
      if ((it != j.end()) && it->is_array()) {
        size_t  i = 0;
        for ( ; i < it->size(); ++i) {
          auto  nameit = (*it)[i].find("qname");
          if ((nameit != (*it)[i].end()) && nameit->is_string()) {
            auto typeit = (*it)[i].find("qtype");
            if ((typeit != (*it)[i].end()) && typeit->is_number()) {
              Dns::MessageQuestion  dnsq(nameit->get<string>(),
                                         typeit->get<uint16_t>(),
                                         Dns::MessageQuestion::k_classIN);
              dnsqs.push_back(dnsq);
            }
            else {
              break;
            }
          }
          else {
            break;
          }
        }
        rc = (it->size() == i);
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool TargetHostConfig::FromJson(const nlohmann::json & j)
    {
      bool  rc = false;
      Clear();
      if (j.is_object()) {
        auto  it = j.find("name");
        if ((it != j.end()) && it->is_string()) {
          _name = it->get<string>();
          it = j.find("address");
          if ((it != j.end()) && it->is_string()) {
            _address = Ipv4Address(it->get<string>());
            rc = true;
          }
          it = j.find("address6");
          if ((it != j.end()) && it->is_string()) {
            _address6 = Ipv6Address(it->get<string>());
            rc = true;
          }
          FromJsonArray(j, "tcp4Ports", _tcp4Ports);
          FromJsonArray(j, "tcp6Ports", _tcp6Ports);
          FromJsonArray(j, "tcp4PortsDenied", _tcp4PortsDenied);
          FromJsonArray(j, "tcp6PortsDenied", _tcp6PortsDenied);
          FromJsonArray(j, "rpcPrograms", _expectedPrograms);
          FromJsonArray(j, "rpc6Programs", _expected6Programs);
          FromJsonArray(j, "smtpPorts", _smtpPorts);
          FromJsonArray(j, "smtp6Ports", _smtpPorts);
          FromJsonArray(j, "ups", _ups);
          ObjsFromJsonArray(j, "webTargets", _webTargets);
          ObjsFromJsonArray(j, "webAppTargets", _webAppTargets);
          ObjsFromJsonArray(j, "dnsResolve", _dnsResolve);
          ObjsFromJsonArray(j, "credencePeerTargets", _credencePeerTargets);
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void TargetHostConfig::Clear()
    {
      _name.clear();
      _address = Ipv4Address();
      _address6 = Ipv6Address();
      _tcp4Ports.clear();
      _tcp6Ports.clear();
      _tcp4PortsDenied.clear();
      _tcp6PortsDenied.clear();
      _expectedPrograms.clear();
      _expected6Programs.clear();
      _smtpPorts.clear();
      _smtp6Ports.clear();
      _ups.clear();
      _webTargets.clear();
      _webAppTargets.clear();
      _dnsResolve.clear();
      _credencePeerTargets.clear();
      return;
    }
    
      
  }  // namespace Mcrover

}  // namespace Dwm
