//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverSMTPUtils.cc 12022 $
// @(#) $Id: DwmMcroverSMTPUtils.cc 12022 2022-05-30 19:03:23Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverSMTPUtils.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::SMTPUtils class implementation
//---------------------------------------------------------------------------

#include <chrono>
#include <map>
#include <memory>
#include <thread>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverTargetHostConfig.hh"
#include "DwmMcroverTcpDstAddr.hh"
#include "DwmMcroverSMTPStateMachine.hh"
#include "DwmMcroverSMTPUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverSMTPUtils.cc 12022 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool SMTPUtils::GetAlerts(const AlertOrigin & origin,
                              const std::vector<TargetHostConfig> & hosts,
                              AlertBowl & alerts,
                              bool clearAtEntry)
    {
      if (clearAtEntry) {
        alerts.Clear();
      }
      
      map<TcpDstAddr, unique_ptr<SMTPStateMachine>>  stateMachines;
      for (const auto & host : hosts) {
        for (auto port : host.SMTPPorts()) {
          stateMachines[TcpDstAddr(host.Name(),host.Address(), port)] =
            make_unique<SMTPStateMachine>(host.Address(), port);
        }
        for (auto port : host.SMTP6Ports()) {
          stateMachines[TcpDstAddr(host.Name(),host.Address6(), port)] =
            make_unique<SMTPStateMachine>(host.Address6(), port);
        }
      }
      if (! stateMachines.empty()) {
        for (auto & sm : stateMachines) {
          sm.second->Start();
        }
        time_t  startTime = time((time_t *)0);
        time_t  endTime = startTime + 5;
        while ((! stateMachines.empty())
               && (time((time_t *)0) < endTime)) {
          for (auto it = stateMachines.begin(); it != stateMachines.end(); ) {
            it->second->Continue();
            if (it->second->Done()) {
              if (it->second->CurrentState() != SMTPPassed::Instance()) {
                alerts.Add(origin, SMTPAlert(it->first),
                           time((time_t *)0));
                Syslog(LOG_ERR, "SMTP failed for %s port %hu",
                       ((string)(it->first.Address())).c_str(),
                       it->first.Port());
              }
              else {
                Syslog(LOG_INFO, "SMTP passed for %s port %hu",
                       ((string)(it->first.Address())).c_str(),
                       it->first.Port());
              }
              it = stateMachines.erase(it++);
            }
            else {
              ++it;
            }
          }
          this_thread::sleep_for(chrono::milliseconds(50));
        }
        for (auto & sm : stateMachines) {
          sm.second->Stop();
          if (sm.second->Done()) {
            if (sm.second->CurrentState() != SMTPPassed::Instance()) {
              alerts.Add(origin, SMTPAlert(sm.first),
                         time((time_t *)0));
              Syslog(LOG_ERR, "SMTP failed for %s port %hu",
                     ((string)(sm.first.Address())).c_str(),
                     sm.first.Port());
            }
            else {
              Syslog(LOG_INFO, "SMTP passed for %s port %hu",
                     ((string)(sm.first.Address())).c_str(),
                       sm.first.Port());
            }
          }
        }
      }
      else {
        Syslog(LOG_INFO, "No SMTP tests configured.");
      }
      return (! alerts.Empty());
    }
    
  }  // namespace Mcrover

}  // namespace Dwm
