//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverSMTPStateMachine.cc 12375 $
// @(#) $Id: DwmMcroverSMTPStateMachine.cc 12375 2024-05-31 19:44:49Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020, 2024
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverSMTPStateMachine.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::SMTPStateMachine class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <fcntl.h>
  #include <unistd.h>

#ifndef MSG_NOSIGNAL
  #define MSG_NOSIGNAL 0
#endif
}

#include <cstring>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverSMTPStateMachine.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverSMTPStateMachine.cc 12375 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    using Clock = std::chrono::system_clock;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string SMTPStateMachine::_noOpString("NOOP\r\n");
    mutex SMTPInitial::_instanceMutex;
    SMTPInitial *SMTPInitial::_instance = nullptr;
    mutex SMTPConnecting::_instanceMutex;
    SMTPConnecting *SMTPConnecting::_instance = nullptr;
    mutex SMTPAwaitingInitialResponse::_instanceMutex;
    SMTPAwaitingInitialResponse *SMTPAwaitingInitialResponse::_instance =
      nullptr;
    mutex SMTPSendingNoOp::_instanceMutex;
    SMTPSendingNoOp *SMTPSendingNoOp::_instance = nullptr;
    mutex SMTPAwaitingNoOpResponse::_instanceMutex;
    SMTPAwaitingNoOpResponse *SMTPAwaitingNoOpResponse::_instance = nullptr;
    mutex SMTPStopped::_instanceMutex;
    SMTPStopped *SMTPStopped::_instance = nullptr;
    mutex SMTPFailed::_instanceMutex;
    SMTPFailed *SMTPFailed::_instance = nullptr;
    mutex SMTPPassed::_instanceMutex;
    SMTPPassed *SMTPPassed::_instance = nullptr;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPState::Start(SMTPStateMachine *sm)       { return; }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPState::Continue(SMTPStateMachine *sm)    { return; }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPState::Stop(SMTPStateMachine *sm)        { return; }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string SMTPState::Name() const                    { return ""; }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    SMTPInitial *SMTPInitial::Instance()
    {
      lock_guard<mutex>  lock(_instanceMutex);
      if (! _instance) {
        _instance = new SMTPInitial();
      }
      return _instance;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPInitial::Start(SMTPStateMachine *sm)
    {
      if (sm->OpenSocket()) {
        sm->ChangeState(SMTPConnecting::Instance());
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPInitial::Continue(SMTPStateMachine *sm)
    {
      if (sm->OpenSocket()) {
        sm->ChangeState(SMTPConnecting::Instance());
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPInitial::Stop(SMTPStateMachine *sm)
    {
      sm->CloseSocket();
      sm->ChangeState(SMTPStopped::Instance());
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    SMTPConnecting *SMTPConnecting::Instance()
    {
      lock_guard<mutex>  lock(_instanceMutex);
      if (! _instance) {
        _instance = new SMTPConnecting();
      }
      return _instance;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPConnecting::Continue(SMTPStateMachine *sm)
    {
      if (sm->Connect()) {
        sm->ChangeState(SMTPAwaitingInitialResponse::Instance());
      }
      else if (sm->TimeInCurrentState() > std::chrono::milliseconds(5000)) {
        sm->ChangeState(SMTPFailed::Instance());
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPConnecting::Stop(SMTPStateMachine *sm)
    {
      sm->CloseSocket();
      sm->ChangeState(SMTPStopped::Instance());
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    SMTPAwaitingInitialResponse *SMTPAwaitingInitialResponse::Instance()
    {
      lock_guard<mutex>  lock(_instanceMutex);
      if (! _instance) {
        _instance = new SMTPAwaitingInitialResponse();
      }
      return _instance;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPAwaitingInitialResponse::Continue(SMTPStateMachine *sm)
    {
      if (sm->InitialResponse().empty()
          || (sm->InitialResponse().back() != '\n')) {
        char  buf[128] = { '\0' };
        errno = 0;
        int   rc = recv(sm->Fd(), buf, 127, 0);
        if (rc > 0) {
          sm->InitialResponse() += buf;
        }
        else if (rc < 0) {
          if (EAGAIN != errno) {
            Syslog(LOG_ERR, "recv(%d, %p, 127) failed: %m",
                   sm->Fd(), buf);
            sm->CloseSocket();
            sm->ChangeState(SMTPFailed::Instance());
          }
        }
      }
      if ((! sm->InitialResponse().empty())
          && (sm->InitialResponse().back() == '\n')) {
        // got response (newline is end of response)
        if (atoi(sm->InitialResponse().c_str()) == 220) {
          //  good response, send NOOP
          Syslog(LOG_DEBUG, "SMTP initial response: '%s'",
                 sm->InitialResponse().c_str());
          sm->ChangeState(SMTPSendingNoOp::Instance());
          sm->Continue();
        }
        else {
          Syslog(LOG_ERR, "SMTP initial response: %s",
                 sm->InitialResponse().c_str());
          sm->CloseSocket();
          sm->ChangeState(SMTPFailed::Instance());
        }
      }

      if (sm->TimeInCurrentState() > std::chrono::milliseconds(5000)) {
        sm->CloseSocket();
        sm->ChangeState(SMTPFailed::Instance());
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPAwaitingInitialResponse::Stop(SMTPStateMachine *sm)
    {
      sm->CloseSocket();
      sm->ChangeState(SMTPStopped::Instance());
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    SMTPSendingNoOp *SMTPSendingNoOp::Instance()
    {
      lock_guard<mutex>  lock(_instanceMutex);
      if (! _instance) {
        _instance = new SMTPSendingNoOp();
      }
      return _instance;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPSendingNoOp::Continue(SMTPStateMachine *sm)
    {
      size_t  bytesSent = sm->NoOpBytesSent();
      if (bytesSent < sm->NoOpString().size()) {
        size_t   bytesRemaining = sm->NoOpString().size() - bytesSent;
        ssize_t  rc = send(sm->Fd(), &(sm->NoOpString()[bytesSent]),
                           bytesRemaining, MSG_NOSIGNAL);
        if (0 < rc) {
          sm->NoOpBytesSent() += rc;
          if (sm->NoOpBytesSent() == sm->NoOpString().size()) {
            sm->ChangeState(SMTPAwaitingNoOpResponse::Instance());
          }
        }
        else if (0 > rc) {
          sm->CloseSocket();
          sm->ChangeState(SMTPFailed::Instance());
        }
        else if (sm->TimeInCurrentState() > std::chrono::milliseconds(5000)) {
          sm->CloseSocket();
          sm->ChangeState(SMTPFailed::Instance());
        }
      }
      else {
        sm->ChangeState(SMTPAwaitingNoOpResponse::Instance());
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPSendingNoOp::Stop(SMTPStateMachine *sm)
    {
      sm->CloseSocket();
      sm->ChangeState(SMTPStopped::Instance());
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    SMTPAwaitingNoOpResponse *SMTPAwaitingNoOpResponse::Instance()
    {
      lock_guard<mutex>  lock(_instanceMutex);
      if (! _instance) {
        _instance = new SMTPAwaitingNoOpResponse();
      }
      return _instance;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPAwaitingNoOpResponse::Continue(SMTPStateMachine *sm)
    {
      size_t  bytesReceived = sm->NoOpResponse().size();
      if (bytesReceived < 4) {
        char  buf[128] = { '\0' };
        ssize_t  rc = recv(sm->Fd(), buf, 127, MSG_DONTWAIT);
        if (rc > 0) {
          sm->NoOpResponse() += buf;
          if (sm->NoOpResponse().size() >= 4) {
            if (atoi(sm->NoOpResponse().c_str()) == 250) {
              sm->CloseSocket();
              sm->ChangeState(SMTPPassed::Instance());
            }
            else {
              sm->CloseSocket();
              sm->ChangeState(SMTPFailed::Instance());
            }
          }
        }
        else if (rc < 0) {
          if (EAGAIN != errno) {
            Syslog(LOG_ERR, "recv(%d, %p, 127) failed: %m",
                   sm->Fd(), buf);
            sm->CloseSocket();
            sm->ChangeState(SMTPFailed::Instance());
          }
        }
        if (sm->TimeInCurrentState() > std::chrono::milliseconds(5000)) {
          //  We've been in current state too long.
          sm->CloseSocket();
          sm->ChangeState(SMTPFailed::Instance());
        }
      }
      else {
        if (atoi(sm->NoOpResponse().c_str()) == 250) {
          sm->CloseSocket();
          sm->ChangeState(SMTPPassed::Instance());
        }
        else {
          sm->CloseSocket();
          sm->ChangeState(SMTPFailed::Instance());
        }
      }
      
      return;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPAwaitingNoOpResponse::Stop(SMTPStateMachine *sm)
    {
      sm->CloseSocket();
      sm->ChangeState(SMTPStopped::Instance());
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    SMTPStopped *SMTPStopped::Instance()
    {
      lock_guard<mutex>  lock(_instanceMutex);
      if (! _instance) {
        _instance = new SMTPStopped();
      }
      return _instance;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    SMTPFailed *SMTPFailed::Instance()
    {
      lock_guard<mutex>  lock(_instanceMutex);
      if (! _instance) {
        _instance = new SMTPFailed();
      }
      return _instance;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    SMTPPassed *SMTPPassed::Instance()
    {
      lock_guard<mutex>  lock(_instanceMutex);
      if (! _instance) {
        _instance = new SMTPPassed();
      }
      return _instance;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    SMTPStateMachine::SMTPStateMachine(const IpAddress & addr, uint16_t port)
        : _addr(addr), _port(port), _fd(-1),
          _currentState(SMTPInitial::Instance()),
          _previousState(_currentState), _lastStateChangeTime(),
          _initialResponse(), _noOpBytesSent(0), _noOpResponse()
    {}

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool SMTPStateMachine::Start()
    {
      bool  rc = false;
      if (_currentState) {
        _currentState->Start(this);
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool SMTPStateMachine::Continue()
    {
      bool  rc = false;
      if (_currentState) {
        _currentState->Continue(this);
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool SMTPStateMachine::Stop()
    {
      bool  rc = false;
      if (_currentState) {
        _currentState->Stop(this);
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPStateMachine::Reset()
    {
      Stop();
      CloseSocket();
      _currentState = SMTPInitial::Instance();
      _initialResponse.clear();
      _noOpBytesSent = 0;
      _noOpResponse.clear();
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool SMTPStateMachine::Done() const
    {
      return ((_currentState == SMTPPassed::Instance())
              || (_currentState == SMTPFailed::Instance())
              || (_currentState == SMTPStopped::Instance()));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool SMTPStateMachine::OpenSocket()
    {
      bool  rc = false;
      _fd = socket(_addr.Family(), SOCK_STREAM, 0);
      if (0 <= _fd) {
        int  fdFlags = fcntl(_fd, F_GETFL, 0) | O_NONBLOCK;
        if (fcntl(_fd, F_SETFL, fdFlags) == 0) {
          rc = true;
        }
        else {
          Syslog(LOG_ERR, "fcntl(%d,F_SETFL,O_NONBLOCK) failed: %m", _fd);
          close(_fd);
          _fd = -1;
        }
      }
      else {
        Syslog(LOG_ERR, "socket(%d,SOCK_STREAM,0) failed: %m",
               _addr.Family());
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool SMTPStateMachine::Connect()
    {
      bool  rc = false;
      if (0 <= _fd) {
        int  connectrc = -1;
        if (_addr.Family() == AF_INET) {
          struct sockaddr_in  sockAddr;
          memset(&sockAddr, 0, sizeof(sockAddr));
          sockAddr.sin_family = _addr.Family();
          sockAddr.sin_addr.s_addr = _addr.Addr<Ipv4Address>()->Raw();
          sockAddr.sin_port = htons(_port);
#ifndef __linux__
          sockAddr.sin_len = sizeof(sockAddr);
#endif
          errno = 0;
          connectrc =
            connect(_fd, (struct sockaddr *)&sockAddr, sizeof(sockAddr));
        }
        else {
          struct sockaddr_in6  sockAddr;
          memset(&sockAddr, 0, sizeof(sockAddr));
          sockAddr.sin6_family = _addr.Family();
          sockAddr.sin6_addr = *(_addr.Addr<Ipv6Address>());
          sockAddr.sin6_port = htons(_port);
#ifndef __linux__
          sockAddr.sin6_len = sizeof(sockAddr);
#endif
          errno = 0;
          connectrc =
            connect(_fd, (struct sockaddr *)&sockAddr, sizeof(sockAddr));
        }
        
        if (0 == connectrc) {
          rc = true;
        }
        else if (EISCONN == errno) {
          rc = true;
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPStateMachine::CloseSocket()
    {
      if (0 <= _fd) {
        if (close(_fd) == 0) {
          _fd = -1;
        }
        else {
          Syslog(LOG_ERR, "close(%d) failed: %m", _fd);
        }
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void SMTPStateMachine::ChangeState(SMTPState *state)
    {
      Syslog(LOG_DEBUG, "%s:%hu SMTP state %s -> %s",
             ((string)_addr).c_str(), _port,
             _currentState->Name().c_str(), state->Name().c_str());
      _previousState = _currentState;
      _currentState = state;
      _lastStateChangeTime = Clock::now();
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Clock::duration SMTPStateMachine::TimeInCurrentState() const
    {
      Clock::time_point  tpnow = Clock::now();
      return Clock::duration(tpnow - _lastStateChangeTime);
    }
    

    
  }  // namespace Mcrover

}  // namespace Dwm
