//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverPackMember.cc 12195 $
// @(#) $Id: DwmMcroverPackMember.cc 12195 2023-06-26 06:20:30Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverPackMember.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::PackMember class implementation
//---------------------------------------------------------------------------

#include <sstream>
#include <boost/asio.hpp>

#include "DwmSocket.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmCredencePeer.hh"
#include "DwmMcroverPackMember.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverPackMember.cc 12195 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    PackMember::PackMember(const PackMemberConfig & config,
                           const string & keyDir)
        : _config(config), _keyDirectory(keyDir)
    {
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const PackMemberConfig & PackMember::MemberConfig() const
    {
      return _config;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const PackMemberConfig &
    PackMember::MemberConfig(const PackMemberConfig & memberConfig)
    {
      _config = memberConfig;
      return _config;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & PackMember::KeyDirectory() const
    {
      return _keyDirectory;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & PackMember::KeyDirectory(const string & keyDir)
    {
      _keyDirectory = keyDir;
      return _keyDirectory;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PackMember::GetMemberAlerts(AlertBowl & alerts,
                                     RoverAlert::FailureType & failure) const
    {
      return GetAlerts(alerts, e_getMemberAlerts, failure);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PackMember::GetPackAlerts(AlertBowl & alerts,
                                   RoverAlert::FailureType & failure) const
    {
      return GetAlerts(alerts, e_getPackAlerts, failure);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PackMember::GetAllAlerts(AlertBowl & alerts,
                                  RoverAlert::FailureType & failure) const
    {
      return GetAlerts(alerts, e_getAllAlerts, failure);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PackMember::GetAlerts(AlertBowl & alerts,
                               McroverdRequest whichAlerts,
                               RoverAlert::FailureType & failure) const
    {
      namespace ip = boost::asio::ip;
      
      bool  rc = false;
      failure = RoverAlert::e_failUnknown;
      alerts.Clear();
      Credence::Peer  peer;
      bool            connected = false;
      for (const auto & packAddr : _config.Addresses()) {
        if (! peer.Connect(packAddr.address().to_string(), packAddr.port())) {
          Syslog(LOG_ERR, "connect(%s) (%s:%hu) failed",
                 _config.Name().c_str(),
                 packAddr.address().to_string().c_str(),
                 packAddr.port());
          peer.Disconnect();
        }
        else {
          Syslog(LOG_INFO, "Connected to pack member %s at %s:%hu",
                 _config.Name().c_str(),
                 packAddr.address().to_string().c_str(),
                 packAddr.port());
          connected = true;
          break;
        }
      }
      if (connected) {
        Credence::KeyStash   keyStash(_keyDirectory);
        Credence::KnownKeys  knownKeys(_keyDirectory);
        if (peer.Authenticate(keyStash, knownKeys)) {
          if (peer.Send((uint8_t &)whichAlerts)) {
            if (peer.Receive(alerts)) {
              rc = true;
              Syslog(LOG_INFO, "Got %lu alerts from pack member %s at %s",
                     alerts.Size(), _config.Name().c_str(),
                     peer.EndPointString().c_str());
            }
            else {
              failure = RoverAlert::e_failRead;
              Syslog(LOG_ERR,
                     "Failed to read alerts from pack member %s at %s",
                     _config.Name().c_str(),
                     peer.EndPointString().c_str());
            }
          }
          else {
            failure = RoverAlert::e_failWrite;
            Syslog(LOG_ERR, "Failed to send alerts to pack member %s at %s",
                   _config.Name().c_str(),
                   peer.EndPointString().c_str());
          }
        }
        else {
          failure = RoverAlert::e_failAuth;
          Syslog(LOG_ERR, "Failed to authenticate to pack member %s at %s",
                 _config.Name().c_str(),
                 peer.EndPointString().c_str());
        }
      }
      else {
        failure = RoverAlert::e_failConnect;
        Syslog(LOG_ERR, "Failed to connect to pack member %s",
               _config.Name().c_str());
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PackMember::GetConfig(Config & cfg) const
    {
      namespace ip = boost::asio::ip;
      
      bool  rc = false;
      Credence::Peer  peer;
      bool            connected = false;
      for (const auto & packAddr : _config.Addresses()) {
        if (! peer.Connect(packAddr.address().to_string(), packAddr.port())) {
          Syslog(LOG_ERR, "connect(%s) (%s:%hu) failed",
                 _config.Name().c_str(),
                 packAddr.address().to_string().c_str(),
                 packAddr.port());
          peer.Disconnect();
        }
        else {
          Syslog(LOG_INFO, "Connected to pack member %s at %s:%hu",
                 _config.Name().c_str(),
                 packAddr.address().to_string().c_str(),
                 packAddr.port());
          connected = true;
          break;
        }
      }
      if (connected) {
        Credence::KeyStash   keyStash(_keyDirectory);
        Credence::KnownKeys  knownKeys(_keyDirectory);
        if (peer.Authenticate(keyStash, knownKeys)) {
          if (peer.Send((const uint8_t &)e_getConfig)) {
            if (peer.Receive(cfg)) {
              rc = true;
              Syslog(LOG_INFO, "Got config from pack member %s at %s",
                     _config.Name().c_str(), peer.EndPointString().c_str());
            }
            else {
              Syslog(LOG_ERR,
                     "Failed to read config from pack member %s at %s",
                     _config.Name().c_str(),
                     peer.EndPointString().c_str());
            }
          }
          else {
            Syslog(LOG_ERR,
                   "Failed to send config request to pack member %s at %s",
                   _config.Name().c_str(),
                   peer.EndPointString().c_str());
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to authenticate to pack member %s at %s",
                 _config.Name().c_str(),
                 peer.EndPointString().c_str());
        }
      }
      else {
        Syslog(LOG_ERR, "Failed to connect to pack member %s",
               _config.Name().c_str());
      }
      return rc;
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm
