//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverAlertBowl.cc 11961 $
// @(#) $Id: DwmMcroverAlertBowl.cc 11961 2022-04-23 17:43:21Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverAlertBowl.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::AlertBowl class implementation
//---------------------------------------------------------------------------

#include <fstream>
#include <mutex>

#include "DwmDateTime.hh"
#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmIOUtils.hh"
#include "DwmStreamIO.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverAlertBowl.hh"
#include "DwmMcroverUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/classes/src/DwmMcroverAlertBowl.cc 11961 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    AlertBowlValue::AlertBowlValue(const AlertBowlValue & abv)
        : _mtx()
    {
      _data = abv._data;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    AlertBowlValue & AlertBowlValue::operator = (const AlertBowlValue & abv)
    {
      if (&abv != this) {
        _data = abv._data;
      }
      return *this;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowlValue::Add(const AlertOrigin & origin, uint64_t t)
    {
      unique_lock  lck(_mtx);
      _data[origin] = t;
      return true;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowlValue::Add(const AlertBowlValue & abv)
    {
      unique_lock  lck(_mtx);
      for (const auto & d : abv._data) {
        if (_data.find(d.first) == _data.end()) {
          _data[d.first] = d.second;
        }
      }
      return (! _data.empty());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowlValue::Remove(const AlertOrigin & origin)
    {
      bool  rc = false;
      unique_lock  lck(_mtx);
      auto  it = _data.find(origin);
      if (it != _data.end()) {
        _data.erase(it);
        rc = true;
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowlValue::RemovePack(const std::string & packName)
    {
      bool  rc = false;
      unique_lock  lck(_mtx);
      for (auto it = _data.begin(); it != _data.end(); ) {
        if (it->first.PackName() == packName) {
          it = _data.erase(it);
          rc = true;
        }
        else {
          ++it;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowlValue::Refresh(const AlertBowlValue & abv)
    {
      for (auto it = _data.begin(); it != _data.end(); ) {
        auto ait = abv._data.find(it->first);
        if (ait == abv._data.end()) {
          it = _data.erase(it);
        }
        else {
          ++it;
        }
      }
      for (auto ait = abv._data.begin(); ait != abv._data.end(); ++ait) {
        auto  it = _data.find(ait->first);
        if (it == _data.end()) {
          _data[ait->first] = ait->second;
        }
      }
      return (! _data.empty());
    }
             
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowlValue::operator == (const AlertBowlValue & abv) const
    {
      scoped_lock  lock(_mtx, abv._mtx);
      return (_data == abv._data);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowlValue::Empty() const
    {
      shared_lock  lck(_mtx);
      return _data.empty();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string AlertBowlValue::DisplayString() const
    {
      std::string  rc;
      std::string  origins;
      shared_lock  lck(_mtx);
      time_t       t = 0;
      if (GetConcise(t, origins)) {
        rc += DateTime(TimeValue64(t, 0)).Formatted("%m/%d %H:%M (");
        rc += origins + ')';
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowlValue::GetConcise(time_t & t, std::string & origins) const
    {
      bool  rc = false;
      origins.clear();
      t = 0;
      shared_lock  lck(_mtx);
      if (! _data.empty()) {
        uint64_t     lt = 0xFFFFFFFFFFFFFFFFULL;
        AlertOrigin  origin;
        for (const auto & d : _data) {
          if (d.second < lt) {
            lt = d.second;
            origin = d.first;
          }
        }
        t = lt;
        if (_data.size() > 1) {
          origins = to_string(_data.size());
        }
        else {
          origins = origin.PackName() + ':' + (string)(origin.MemberAddress());
        }
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowlValue::GetByPack(const string & packName,
                                   AlertBowlValue & abv) const
    {
      abv._data.clear();
      shared_lock  lck(_mtx);
      if (! _data.empty()) {
        for (const auto & d : _data) {
          if (d.first.PackName() == packName) {
            abv._data[d.first] = d.second;
          }
        }
      }
      return (! abv._data.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowlValue::GetByOrigin(const AlertOrigin & origin,
                                     AlertBowlValue & abv) const
    {
      abv._data.clear();
      shared_lock  lck(_mtx);
      auto it = _data.find(origin);
      if (it != _data.end()) {
        abv._data[it->first] = it->second;
      }
      return (! abv._data.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::pair<AlertOrigin,uint64_t> AlertBowlValue::Oldest() const
    {
      std::pair<AlertOrigin,uint64_t>  rc;
      shared_lock  lck(_mtx);
      if (_data.size() == 1) {
        rc = *(_data.begin());
      }
      else {
        if (! _data.empty()) {
          auto  guess = _data.begin();
          auto  it = guess;
          ++it;
          for ( ; it != _data.end(); ++it) {
            if (it->second < guess->second) {
              guess = it;
            }
          }
          rc = *guess;
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::pair<AlertOrigin,uint64_t> AlertBowlValue::Newest() const
    {
      std::pair<AlertOrigin,uint64_t>  rc;
      shared_lock  lck(_mtx);
      if (_data.size() == 1) {
        rc = *(_data.begin());
      }
      else {
        if (! _data.empty()) {
          auto  guess = _data.begin();
          auto  it = guess;
          ++it;
          for ( ; it != _data.end(); ++it) {
            if (it->second > guess->second) {
              guess = it;
            }
          }
          rc = *guess;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & operator << (std::ostream & os, const AlertBowlValue & abv)
    {
      for (const auto d : abv._data) {
        os << "  " << d.first.PackName() << ':' << d.first.MemberAddress()
           << '\n';
      }
      return os;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & AlertBowlValue::Read(std::istream & is)
    {
      unique_lock  lck(_mtx);
      return Dwm::StreamIO::Read(is, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t AlertBowlValue::Read(int fd)
    {
      unique_lock  lck(_mtx);
      return Dwm::DescriptorIO::Read(fd, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t AlertBowlValue::Read(FILE *f)
    {
      unique_lock  lck(_mtx);
      return Dwm::FileIO::Read(f, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & AlertBowlValue::Write(std::ostream & os) const
    {
      shared_lock  lck(_mtx);
      return Dwm::StreamIO::Write(os, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t AlertBowlValue::Write(int fd) const
    {
      shared_lock  lck(_mtx);
      return Dwm::DescriptorIO::Write(fd, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t AlertBowlValue::Write(FILE *f) const
    {
      shared_lock  lck(_mtx);
      return Dwm::FileIO::Write(f, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t AlertBowlValue::StreamedLength() const
    {
      shared_lock  lck(_mtx);
      return Dwm::IOUtils::StreamedLength(_data);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::Load(const std::string & fileName)
    {
      bool  rc = false;
      ifstream  is(fileName.c_str());
      if (is) {
        if (Read(is)) {
          rc = true;
        }
        else {
          Syslog(LOG_ERR, "Failed to read alerts from '%s'", fileName.c_str());
        }
      }
      else {
        Syslog(LOG_ERR, "Failed to open '%s'", fileName.c_str());
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::Save(const std::string & fileName) const
    {
      bool  rc = false;
      ofstream  os(fileName.c_str());
      if (os) {
        if (Write(os)) {
          rc = true;
        }
        else {
          Syslog(LOG_ERR, "Failed to write alerts to '%s'", fileName.c_str());
        }
      }
      else {
        Syslog(LOG_ERR, "Failed to open '%s'", fileName.c_str());
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::Add(const AlertOrigin & origin, const KeyType & key,
                        uint64_t t)
    {
      unique_lock  lck(_mtx);
      return _data[key].Add(origin, t);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::Add(const AlertBowl & alerts)
    {
      unique_lock  lck(_mtx);
      for (const auto & al : alerts._data) {
        _data[al.first].Add(al.second);
      }
      return (! _data.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::Remove(const AlertOrigin & origin, const KeyType & key)
    {
      bool  rc = false;
      unique_lock  lck(_mtx);
      auto  it = _data.find(key);
      if (it != _data.end()) {
        if (it->second.Remove(origin)) {
          rc = true;
          if (it->second.Empty()) {
            _data.erase(it);
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::RemoveOrigin(const AlertOrigin & origin)
    {
      bool  rc = false;
      unique_lock  lck(_mtx);
      for (auto it = _data.begin(); it != _data.end(); ) {
        if (it->second.Remove(origin)) {
          rc = true;
          if (it->second.Empty()) {
            it = _data.erase(it);
          }
          else {
            ++it;
          }
        }
        else {
          ++it;
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::RemovePack(const std::string & packName)
    {
      bool  rc = false;
      unique_lock  lck(_mtx);
      for (auto it = _data.begin(); it != _data.end(); ) {
        if (it->second.RemovePack(packName)) {
          rc = true;
          if (it->second.Empty()) {
            it = _data.erase(it);
          }
          else {
            ++it;
          }
        }
        else {
          ++it;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void AlertBowl::Refresh(const AlertBowl & alerts)
    {
      for (auto bi = _data.begin(); bi != _data.end(); ) {
        auto  ai = alerts._data.find(bi->first);
        if (ai == alerts._data.end()) {
          bi = _data.erase(bi);
        }
        else {
          ++bi;
        }
      }
      for (auto ai = alerts._data.begin(); ai != alerts._data.end(); ++ai) {
        auto bi = _data.find(ai->first);
        if (bi != _data.end()) {
          bi->second.Refresh(ai->second);
        }
        else {
          _data[ai->first].Add(ai->second);
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::Empty() const
    {
      shared_lock  lck(_mtx);
      return _data.empty();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t AlertBowl::Size() const
    {
      shared_lock  lck(_mtx);
      return _data.size();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void AlertBowl::Clear()
    {
      unique_lock  lck(_mtx);
      _data.clear();
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::GetConcise(vector<pair<time_t,string>> & alerts) const
    {
      bool  rc = false;
      alerts.clear();
      shared_lock  lck(_mtx);
      for (const auto & al : _data) {
        time_t  t;
        string  orig;
        if (al.second.GetConcise(t, orig)) {
          string  origAndAlert("(");
          origAndAlert += orig + ") ";
          std::visit([&origAndAlert](auto const & p)
                     { origAndAlert += p.DisplayString(); }, al.first);
          alerts.push_back({ t, origAndAlert });
        }
      }
      return (! alerts.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void AlertBowl::Get(vector<pair<KeyType,AlertBowlValue>> & alerts) const
    {
      alerts.clear();
      for (const auto & al : _data) {
        AlertBowlValue  abv = al.second;
        alerts.push_back(make_pair<KeyType,AlertBowlValue>(KeyType(al.first),std::move(abv)));
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::GetByPack(const string & packName,
                              AlertBowl & alerts) const
    {
      alerts.Clear();
      shared_lock  lck(_mtx);
      for (auto it = _data.begin(); it != _data.end(); ++it) {
        AlertBowlValue  abv;
        it->second.GetByPack(packName, abv);
        if (! abv.Empty()) {
          alerts._data.insert(make_pair<const KeyType, AlertBowlValue>(KeyType(it->first),
                                                                       std::move(abv)));
        }
      }
      return (! alerts._data.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::GetByOrigin(const AlertOrigin & origin,
                                AlertBowl & alertBowl) const
    {
      alertBowl.Clear();
      shared_lock  lck(_mtx);
      for (auto it = _data.begin(); it != _data.end(); ++it) {
        AlertBowlValue  abv;
        if (it->second.GetByOrigin(origin, abv)) {
          alertBowl._data.insert(make_pair<const KeyType, AlertBowlValue>(KeyType(it->first),
                                                                          std::move(abv)));
        }
      }
      return (! alertBowl._data.empty());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & operator << (std::ostream & os, const AlertBowl & ab)
    {
      for (const auto & d : ab._data) {
        std::visit([&os](auto const & p)
        { os << p.DisplayString() << '\n'; }, d.first);
        os << d.second;
      }
      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & AlertBowl::Read(std::istream & is)
    {
      unique_lock  lck(_mtx);
      return StreamIO::Read(is, _data);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t AlertBowl::Read(int fd)
    {
      unique_lock  lck(_mtx);
      return DescriptorIO::Read(fd, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t AlertBowl::Read(FILE * f)
    {
      unique_lock  lck(_mtx);
      return FileIO::Read(f, _data);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & AlertBowl::Write(std::ostream & os) const
    {
      shared_lock  lck(_mtx);
      return StreamIO::Write(os, _data);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t AlertBowl::Write(int fd) const
    {
      shared_lock  lck(_mtx);
      return DescriptorIO::Write(fd, _data);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t AlertBowl::Write(FILE *f) const
    {
      shared_lock  lck(_mtx);
      return FileIO::Write(f, _data);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t AlertBowl::StreamedLength() const
    {
      shared_lock  lck(_mtx);
      return IOUtils::StreamedLength(_data);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AlertBowl::operator == (const AlertBowl & ab) const
    {
      scoped_lock  lock(_mtx, ab._mtx);
      return (_data == ab._data);
    }
    
      
  }  // namespace Mcrover

}  // namespace Dwm
