//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/apps/qmcrover/DwmMcroverWeatherImageWidget.cc 11932 $
// @(#) $Id: DwmMcroverWeatherImageWidget.cc 11932 2022-04-11 02:01:48Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverWeatherImageWidget.cc
//!  \author Daniel W. McRobb
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#include <cmath>

#include <QBrush>
#include <QImage>
#include <QPen>

#include "DwmSvnTag.hh"
#include "DwmMcroverWeatherImageWidget.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/apps/qmcrover/DwmMcroverWeatherImageWidget.cc 11932 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    /*
      "shortForecast" examples:
          "Chance Rain And Snow Showers then Mostly Cloudy",
          "Partly Sunny",
          "Mostly Sunny",
          "Sunny",
          "Mostly Clear",  (night only?)
          "Clear", (night only?)
          "Partly Cloudy",
          "Mostly Cloudy",
          "Chance Light Rain",
          "Slight Chance Rain",
          "Chance Rain",
          "Rain Likely",
          "Chance Light Rain then Slight Chance Rain And Snow",
          "Slight Chance Rain And Snow",
          "Slight Chance Rain then Partly Sunny"

      Note that these are kinda shitty for machine processing.  Yes, I
      could write a parser with bison/flex, buit my guess is that the NWS
      could changes these at any time and break things.  On the other hand,
      the detail data is MUCH larger to pull down.

      So for day conditions I've seen:
        "Sunny"
        "Mostly Sunny"
        "Partly Sunny"
        "Partly Cloudy"
        "Mostly Cloudy"

      Precipitations:
        "Light Rain"
        "Rain"
        "Rain Showers"
        "Snow"
        "Snow Showers"

      Precip forecast:
        "Slight Chance" Precipitation
        Chance
        Likely

      precipForecast: CHANCE precips
        | SLIGHT CHANCE precips
        | precips LIKELY
        | precipForecast THEN precipForecast;
      precips: precip | precips AND precip;
      precip: LightRain | Rain | RainShowers | Snow | SnowShowers;
      coverage: Slight | Chance | Likely;
      
    */
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    WeatherImageWidget::WeatherImageWidget(QWidget *parent,
                                           Qt::WindowFlags f)
        : QWidget(parent, f)
    {
      setMinimumSize(QSize(120,120));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WeatherImageWidget::Populate(const Mcweather::PeriodForecast & periodForecast)
    {
      _isDaytime = periodForecast.IsDayTime();
      vector<vector<Weather::ImageData>>  wid;
      if (ParseShortForecast(periodForecast.ShortForecast(), wid)) {
        if (! wid.empty()) {
          if (! wid.back().empty()) {
            _imageData = wid.back().back();
          }
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WeatherImageWidget::paintEvent(QPaintEvent *event)
    {
      QPainter  painter(this);
      if (_isDaytime) {
        DrawDaySky(painter);
        if (_imageData.cloudLevel <= 3) {
          DrawSun(painter, QPointF(width() / 2.0, height() / 2.0), 1.0);
        }
        if (_imageData.rainLevel >= 4) {
          QImage  img(":/resources/images/thundercloud.svg");
          DrawImage(painter,
                    ImgLocScaleTuple(img,
                                     QPointF(width() / 2.0, height() / 2.0),
                                     0.9));
        }
        else if (_imageData.rainLevel > 0) {
          QImage  img(":/resources/images/raincloud1.svg");
          DrawImage(painter,
                    ImgLocScaleTuple(img,
                                     QPointF(width() / 2.0, height() / 2.0),
                                     sqrt(_imageData.precipChance / 4.0) * 0.9));
        }
        else {
          DrawCloud(painter);
          if (_imageData.snowLevel > 0) {
            QImage  img(":/resources/images/snowflake1.svg");
            DrawImage(painter,
                    ImgLocScaleTuple(img,
                                     QPointF(width() / 2.0, height() / 2.0),
                                     sqrt(_imageData.precipChance / 4.0) * 0.9));
          }
        }
      }
      else {
        DrawNightSky(painter);
        if (_imageData.cloudLevel <= 3) {
          QImage  img(":/resources/images/moonstar.svg");
          DrawImage(painter,
                    ImgLocScaleTuple(img,
                                     QPointF(width() / 2.0, height() / 2.0),
                                     0.7));
        }
        if (_imageData.rainLevel >= 4) {
          QImage  img(":/resources/images/thundercloud.svg");
          DrawImage(painter,
                    ImgLocScaleTuple(img,
                                     QPointF(width() / 2.0, height() / 2.0),
                                     0.9));
        }
        else if (_imageData.rainLevel > 0) {
          QImage  img(":/resources/images/raincloud1.svg");
          DrawImage(painter,
                    ImgLocScaleTuple(img,
                                     QPointF(width() / 2.0, height() / 2.0),
                                     sqrt(_imageData.precipChance / 4.0) * 0.9));
        }
        else {
          DrawCloud(painter);
          if (_imageData.snowLevel > 0) {
            QImage  img(":/resources/images/snowflake1.svg");
            DrawImage(painter,
                    ImgLocScaleTuple(img,
                                     QPointF(width() / 2.0, height() / 2.0),
                                     sqrt(_imageData.precipChance / 4.0) * 0.9));
          }
        }
      }
      
      // DrawCloud(painter, QPointF(width() / 4.0, height() * 1.0 / 4.0), 0.33);
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WeatherImageWidget::DrawSun(QPainter & painter, QPointF center,
                                     double scale)
    {
      QImage  img(":/resources/images/sun.svg");
      DrawImage(painter, ImgLocScaleTuple(img, center, scale));
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WeatherImageWidget::DrawCloud(QPainter & painter)
    {
      QImage  cloudImage;
      if (_imageData.precipChance == 0) {
        double  scale = sqrt(_imageData.cloudLevel / 5.0);
        QPointF  center(width() / 2.0, height() / 2.5);
        cloudImage = QImage(":/resources/images/wtcloud.svg");
        DrawImage(painter, ImgLocScaleTuple(cloudImage, center, scale));
      }
      else {
        double  scale = sqrt(_imageData.cloudLevel / 5.0);
        QPointF  center(width() / 2.0, height() / 2.5);
        cloudImage = QImage(":/resources/images/dkcloud.svg");
        DrawImage(painter, ImgLocScaleTuple(cloudImage, center, scale));
      }
      
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WeatherImageWidget::DrawCloud(QPainter & painter, QPointF center,
                                       double scale)
    {
      // QImage  img(":/resources/images/ltcloud.svg");
      QImage  img(":/resources/images/wtcloud.svg");
      DrawImage(painter, ImgLocScaleTuple(img, center, scale));
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WeatherImageWidget::DrawImages(QPainter & painter,
                                        const vector<ImgLocScaleTuple> & imgs)
    {
      for (const auto & tup : imgs) {
        DrawImage(painter, tup);
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WeatherImageWidget::
    DrawImage(QPainter & painter,
              const WeatherImageWidget::ImgLocScaleTuple & imgLocScale)
    {
      auto [ img, pos, scale ] = imgLocScale;
      double  hratio = (double)width() / img.width();
      double  vratio = (double)height() / img.height();
      QImage  scaledImage;
      if (hratio < vratio) {
        scaledImage =
          img.scaledToWidth(scale * width(), Qt::SmoothTransformation);
      }
      else {
        scaledImage =
          img.scaledToHeight(scale * height(), Qt::SmoothTransformation);
      }
      pos -= QPointF(scaledImage.width() / 2, scaledImage.height() / 2);
      painter.drawImage(pos, scaledImage);
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WeatherImageWidget::DrawDaySky(QPainter & painter)
    {
      if (_imageData.cloudLevel <= 3) {
        DrawBackground(painter, QColor(0, 170, 228));
      }
      else {
        DrawBackground(painter, QColor(128, 128, 128));
      }
      
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WeatherImageWidget::DrawNightSky(QPainter & painter)
    {
      DrawBackground(painter, QColor(25, 25, 112));
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WeatherImageWidget::DrawBackground(QPainter & painter,
                                            const QColor & bgcolor)
    {
      QBrush  brush(bgcolor);
      QPen    pen(brush, 1);
      painter.setPen(pen);
      painter.setBrush(brush);
      painter.drawRect(0, 0, width(), height());
      return;
    }
    
    
  }  // namespace Mcrover
  
}  // namespace Dwm
