//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/apps/qmcrover/DwmMcroverWeatherFetcher.cc 12361 $
// @(#) $Id: DwmMcroverWeatherFetcher.cc 12361 2024-05-21 22:32:47Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020, 2022
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverWeatherFetcher.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::WeatherFetcher class implementation
//---------------------------------------------------------------------------

#include <fstream>
#include <sstream>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmCredencePeer.hh"
#include "DwmWebUtils.hh"
#include "DwmMcweatherWeatherdRequest.hh"
#include "DwmMcroverUtils.hh"
#include "DwmMcroverWeatherFetcher.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/apps/qmcrover/DwmMcroverWeatherFetcher.cc 12361 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    WeatherFetcher::WeatherFetcher(const Config & config)
        : QObject(), _config(config), _run(false), _runmtx(),
          _runcv(), _thread(), _lastFetchedForecast(0)
    {
      qRegisterMetaType<nlohmann::json>("nlohmann::json");
      qRegisterMetaType<Mcweather::CurrentConditions>("Dwm::Mcrover::Weather::CurrentConditions");
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    WeatherFetcher::~WeatherFetcher()
    {
      Stop();
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool WeatherFetcher::Start()
    {
      bool  rc = false;
      _run = true;
      try { 
        _thread = std::thread(&WeatherFetcher::Run, this);
        rc = true;
      }
      catch (...) {
        Syslog(LOG_ERR, "Exception starting WeatherFetcher thread");
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool WeatherFetcher::Stop()
    {
      {
        lock_guard<mutex>  lk(_runmtx);
        _run = false;
      }
      _runcv.notify_one();
      if (_thread.joinable()) {
        _thread.join();
        Syslog(LOG_INFO, "WeatherFetcher thread joined.");
      }
      return true;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool WeatherFetcher::GetForecasts(Credence::Peer & weatherd)
    {
      bool  rc = false;
      uint8_t  req = Mcweather::e_dailyForecasts;
      if (weatherd.Send(req)) {
        Mcweather::PeriodForecasts  forecasts;
        if (weatherd.Receive(forecasts)) {
          emit newForecasts(forecasts);
          rc = true;
        }
        else {
          Syslog(LOG_ERR, "Failed to receive daily forecasts");
        }
      }
      else {
        Syslog(LOG_ERR, "Failed to send daily forecast request");
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool WeatherFetcher::GetCurrentConditions(Credence::Peer & weatherd)
    {
      bool  rc = false;
      uint8_t  req = Mcweather::e_currentConditions;
      if (weatherd.Send(req)) {
        map<string,Mcweather::CurrentConditions>  cc;
        if (weatherd.Receive(cc)) {
          vector<Mcweather::CurrentConditions>  ccv;
          for (const auto & c : cc) {
            ccv.push_back(c.second);
          }
          if (! ccv.empty()) {
            std::sort(ccv.begin(), ccv.end(),
                      [&] (const Mcweather::CurrentConditions & a,
                           const Mcweather::CurrentConditions & b)
                      { return (a.Timestamp() > b.Timestamp()); });
            emit currentConditions(0, ccv[0]);
            if (ccv.size() > 1) {
              emit currentConditions(1, ccv[1]);
            }
            rc = true;
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to receive current conditions");
        }
      }
      else {
        Syslog(LOG_ERR, "Failed to send current conditions request");
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void WeatherFetcher::Run()
    {
      bool      keepRunning = true;
      uint32_t  cycles = 0;
      
      Syslog(LOG_INFO, "WeatherFetcher thread started");
      while (keepRunning) {
        const WeatherConfig  & wcfg = _config.Weather();
        Credence::Peer  weatherd;
        if (weatherd.Connect(wcfg.Host(), wcfg.Port())) {
          Credence::KeyStash   keyStash;
          Credence::KnownKeys  knownKeys;
          if (weatherd.Authenticate(keyStash, knownKeys)) {
            if (cycles % 4 == 0) {
              if (GetForecasts(weatherd)) {
                _lastFetchedForecast = time((time_t *)0);
              }
            }
            GetCurrentConditions(weatherd);
            uint8_t  req = Mcweather::e_buhBye;
            weatherd.Send(req);
          }
          weatherd.Disconnect();
        }

        {  //  scoped locking
          unique_lock<mutex>  lk(_runmtx);
          if (_runcv.wait_for(lk, 900s) != std::cv_status::timeout) {
            keepRunning = _run;
          }
        }
        ++cycles;
      }
      Syslog(LOG_INFO, "WeatherFetcher thread done");
      
      return;
    }
    

  }  // namespace Mcrover

}  // namespace Dwm
