//===========================================================================
// @(#) $Name:$
// @(#) $Id:$
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020, 2022, 2024
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverNWSRadarPageWidget.cc
//!  \brief Dwm::Mcrover::NWSRadarPageWidget implementation
//---------------------------------------------------------------------------

#include <QFile>
#include <QTimer>
#include <QVBoxLayout>
#include <QWebEngineProfile>

#include "DwmSysLogger.hh"
#include "DwmMcroverNWSRadarPageWidget.hh"

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    NWSRadarPageWidget::NWSRadarPageWidget(QWidget *parent,
                                           Qt::WindowFlags f)
        : QFrame(parent, f), _loaded(false)
    {
      QFile  file;
      file.setFileName("/usr/share/javascript/jquery/jquery.min.js");
      if (file.open(QIODevice::ReadOnly)) {
        _jQuery = file.readAll();
        _jQuery.append("\nvar qt = { 'jQuery': jQuery.noConflict(true) };");
        file.close();
      }
      _webEngView = new QWebEngineView();
      connect(_webEngView, &QWebEngineView::loadFinished,
              this, &NWSRadarPageWidget::finishLoading);
      connect(_webEngView, &QWebEngineView::renderProcessTerminated,
              [=]()
              { QTimer::singleShot(0, this,
                                   &NWSRadarPageWidget::ReloadPage); });
      QVBoxLayout  *layout = new QVBoxLayout();
      setLayout(layout);
      layout->addWidget(_webEngView);
      layout->setSizeConstraint(QLayout::SetMinimumSize);
      layout->setStretchFactor(_webEngView, 1);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void NWSRadarPageWidget::ReloadPage()
    {
      _webEngView->reload();
      TweakPage(this);
    }
    
#if 0
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void
    NWSRadarPageWidget::lifeCycleChange(QWebEnginePage::LifecycleState state)
    {
      if (QWebEnginePage::LifecycleState::Active == state) {
        TweakPage(this);
      }
      return;
    }
#endif
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void NWSRadarPageWidget::finishLoading(bool ok)
    {
      _webEngView->page()->runJavaScript(_jQuery);
      if (ok) {
        TweakPage(this);
      }
      else {
        Syslog(LOG_ERR, "Failed to run javascript in NWSRadarPageWidget");
      }
      
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void NWSRadarPageWidget::SetURL(const std::string & radarURL)
    {
      if (! _loaded) {
        _webEngView->load(QUrl(radarURL.c_str()));
#if 0
        qDebug() << "cookies:"
                 << _webEngView->page()->profile()->persistentCookiesPolicy();
        qDebug() << "cache:" << _webEngView->page()->profile()->cachePath();
        qDebug() << "storage:" << _webEngView->page()->profile()->persistentStoragePath();
        qDebug() << "off the record:" << _webEngView->page()->profile()->isOffTheRecord();
#endif
        _loaded = true;
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void NWSRadarPageWidget::TweakPage(QWidget *page)
    {
      if (dynamic_cast<QWidget *>(this->parent()) == page) {
        // Remove the long title from the header.
        QString code = QStringLiteral("qt.jQuery('.title-long').remove()");
        _webEngView->page()->runJavaScript(code);

        //  Move the radar menu, controls and timeline into the banner and
        //  tweak.
        code = QStringLiteral("qt.jQuery('.cmi-radar-menu-item').detach().appendTo('.banner');"
                              "qt.jQuery('.cmi-radar-menu-item').css('margin-left','20px');"
                              "qt.jQuery('.controls').detach().appendTo('.banner');"
                              "qt.jQuery('.controls').css('margin-left','20px');"
                              "qt.jQuery('.timeline').detach().appendTo('.banner');"
                              "qt.jQuery('.timeline').css('margin-left','20px');"
                              "qt.jQuery('.timeline').css('padding-left','20px');"
                              "qt.jQuery('.timeline').css('padding-right','20px');"
                              "qt.jQuery('.timeline').css('border-radius','9px');");
        _webEngView->page()->runJavaScript(code);

        //  Move/style the legend.
        code = QStringLiteral("qt.jQuery('.legend').detach().appendTo('.banner');"
                              " qt.jQuery('.legend').css('position','absolute');"
                              " qt.jQuery('.legend').css('right','0px');");
        _webEngView->page()->runJavaScript(code);
        
        //  Style the zoom-in button.
        code = QStringLiteral("qt.jQuery('.control-item.fa-plus-circle').css('background','#ff7f2a'); "

                              "qt.jQuery('.control-item.fa-plus-circle').css('color','black'); "
                              "qt.jQuery('.control-item.fa-plus-circle').css('font-size','2em')");
        _webEngView->page()->runJavaScript(code);
        
        //  Style the zoom-out button.
        code = QStringLiteral("qt.jQuery('.control-item.fa-minus-circle').css('background','#ff7f2a'); "
                              "qt.jQuery('.control-item.fa-minus-circle').css('color','black'); "
                              "qt.jQuery('.control-item.fa-minus-circle').css('font-size','2em');");
        _webEngView->page()->runJavaScript(code);
        
        //  Style the info div (time of last update).
        code = QStringLiteral("qt.jQuery('.control-item.info').css('background','black'); "

                              "qt.jQuery('.control-item.info').css('font-size','16px'); "
                              "qt.jQuery('.control-item.info').css('line-height','16px'); "
                              "qt.jQuery('.control-item.info').css('padding-top','4px'); "
                              "qt.jQuery('.control-item.info').css('width','90px')");
        _webEngView->page()->runJavaScript(code);

        //  Hit the play button.
        code = QStringLiteral("qt.jQuery('.button-play').click();");
        _webEngView->page()->runJavaScript(code);
        
        Syslog(LOG_DEBUG, "Radar page tweaks done.");
      }
      return;
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm
