//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.12/apps/mcroverd/DwmMcroverResponder.cc 12195 $
// @(#) $Id: DwmMcroverResponder.cc 12195 2023-06-26 06:20:30Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020, 2022
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverResponder.cc
//!  \author Daniel W. McRobb
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <arpa/inet.h>
}

#include <thread>
#include <boost/asio.hpp>

#include "DwmSysLogger.hh"
#include "DwmCredencePeer.hh"
#include "DwmMcroverAlertBowl.hh"
#include "DwmMcroverServer.hh"

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Responder::Responder(boost::asio::ip::tcp::socket && s, Server & server,
                         const AlertOrigin & origin)
        :  _peer(), _server(server), _origin(origin), _running(false)
    {
      boost::system::error_code  ec;
      s.native_non_blocking(false, ec);
      _peer.SetKeyExchangeTimeout(chrono::milliseconds(2000));
      _peer.SetIdExchangeTimeout(chrono::milliseconds(2000));
      if (_peer.Accept(std::move(s))) {
        _thread = thread(&Responder::Run, this);
        _running.store(true);
      }
      else {
        _peer.Disconnect();
        Syslog(LOG_ERR, "Failed to accept peer");
      }
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Responder::~Responder()
    {
      Join();
      Syslog(LOG_INFO, "Responder for %s destroyed",
             _peer.EndPointString().c_str());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::Join()
    {
      bool  rc = false;
      bool  isRunning = _running.load();
      if (! isRunning) {
        if (_thread.joinable()) {
          _thread.join();
          rc = true;
          Syslog(LOG_DEBUG, "Responder thread for %s joined",
                 _peer.EndPointString().c_str());
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::SendAlerts(const AlertBowl & alerts)
    {
      bool  rc = false;
      if (_peer.Send(alerts)) {
        rc = true;
        Syslog(LOG_INFO, "Sent %lu alerts to %s at %s", alerts.Size(),
               _peer.Id().c_str(), _peer.EndPointString().c_str());
      }
      else {
        Syslog(LOG_ERR, "Failed to send alerts to %s at %s",
               _peer.Id().c_str(), _peer.EndPointString().c_str());
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::SendMyAlerts()
    {
      bool  rc = false;
      AlertBowl  alertBowl;
      if (alertBowl.Load(_server.AlertFile())) {
        AlertBowl  myAlerts;
        alertBowl.GetByOrigin(_origin, myAlerts);
        rc = SendAlerts(myAlerts);
      }
      else {
        Syslog(LOG_ERR, "Failed to load alerts from '%s'",
               _server.AlertFile().c_str());
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::SendMyPackAlerts()
    {
      bool  rc = false;
      AlertBowl  alertBowl;
      if (alertBowl.Load(_server.AlertFile())) {
        AlertBowl  packAlerts;
        alertBowl.GetByPack(_origin.PackName(), packAlerts);
        rc = SendAlerts(packAlerts);
      }
      else {
        Syslog(LOG_ERR, "Failed to load alerts from '%s'",
               _server.AlertFile().c_str());
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::SendAllAlerts()
    {
      bool  rc = false;
      AlertBowl  alertBowl;
      if (alertBowl.Load(_server.AlertFile())) {
        rc = SendAlerts(alertBowl);
      }
      else {
        Syslog(LOG_ERR, "Failed to load alerts from '%s'",
               _server.AlertFile().c_str());
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::SendConfig()
    {
      bool  rc = false;
      rc = _peer.Send(_server.GetConfig());
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::HandleCommand(McroverdRequest req)
    {
      bool  rc = false;
      AlertBowl  alertBowl;

      Syslog(LOG_INFO, "Got req %hhu from %s at %s",
             req, _peer.Id().c_str(), _peer.EndPointString().c_str());
      switch (req) {
        case e_getMemberAlerts:   rc = SendMyAlerts();          break;
        case e_getPackAlerts:     rc = SendMyPackAlerts();      break;
        case e_getAllAlerts:      rc = SendAllAlerts();         break;
        case e_getConfig:         rc = SendConfig();            break;
        default:                                                break;
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Responder::Run()
    {
      Syslog(LOG_INFO, "Responder started for %s",
             _peer.EndPointString().c_str());
      if (_peer.Authenticate(_server.GetKeyStash(), _server.GetKnownKeys())) {
        McroverdRequest  req;
        if (_peer.Receive((uint8_t &)req)) {
          if (! HandleCommand(req)) {
            Syslog(LOG_ERR, "Failed to handle request %hu from %s",
                   req, _peer.EndPointString().c_str());
          }
        }
      }
      _peer.Disconnect();
      Syslog(LOG_DEBUG, "Responder done for %s",
             _peer.EndPointString().c_str());
      
      _running.store(false);
      return;
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm
