//===========================================================================
// @(#) $Name:$
// @(#) $Id: mcloc.cc 10338 2020-02-06 02:38:28Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2019, 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file mcloc.cc
//!  \brief count lines of code in file(s) and/or directories
//---------------------------------------------------------------------------

extern "C" {
  #include <unistd.h>
}

#include <cstdlib>
#include <regex>

#include "DwmMclocConfig.hh"
#include "DwmMclocEnvToArgs.hh"
#include "DwmMclocSourceCollection.hh"
#include "DwmMclocFileScanner.hh"

using namespace std;

static const string g_svnid("@(#) $DwmPath: dwm/mcplex/mcloc/tags/mcloc-1.0.0/mcloc.cc 10338 $");

static const string g_defaultConf("/usr/local/etc/mcloc.cfg");

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void KeepStrings(const std::string & s, set<string> & ss,
                        bool toLowerCase = false)
{
  using namespace Dwm::Mcloc;
  
  ss.clear();
  std::regex  rgx("([^ ,]+)", regex::ECMAScript|regex::optimize);
  auto  vbeg = sregex_iterator(s.begin(), s.end(), rgx);
  auto  vend = sregex_iterator();
  for (auto it = vbeg; it != vend; ++it) {
    ss.insert(toLowerCase ? CodeUtils::tolower(it->str()) : it->str());
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static ostream & PrintLanguagesSupported(ostream & os)
{
  using namespace Dwm::Mcloc;
  
  Config::Config  config;
  if (config.Parse(g_defaultConf)) {
    vector<string>  languages;
    for (const auto & scanner : config.Scanners()) {
      for (const auto & language : scanner.Languages()) {
        languages.push_back(language.Name());
      }
    }
    sort(languages.begin(), languages.end(),
         [] (const string & l1, const string & l2)
         { return (CodeUtils::tolower(l1) < CodeUtils::tolower(l2)); });
    os << "                    ";
    int  col = 20;
    string  sep;
    for (const auto & language : languages) {
      size_t  w = language.size() + 2;
      if ((col + w) < 76) {
        os << sep << language;
        col += w;
      }
      else {
        os << sep << "\n                    " << language;
        col = 20 + w;
      }
      sep = ", ";
    }
    os << '\n';
  }
  return os;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static ostream & PrintScannersSupported(ostream & os)
{
  Dwm::Mcloc::Config::Config  config;
  if (config.Parse(g_defaultConf)) {
    os << "                    ";
    int  col = 20;
    string  sep;
    for (const auto & scanner : config.Scanners()) {
      size_t  w = scanner.Name().size() + 2;
      if ((col + w) < 76) {
        os << sep << scanner.Name();
        col += w;
      }
      else {
        os << sep << "\n                    " << scanner.Name();
        col = 20 + w;
      }
      sep = ", ";
    }
    os << '\n';
  }
  return os;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void Usage(const char *argv0, int numThreads)
{
  cerr << "\nUsage: " << argv0
       << " [-C] [-c config] [-d|-e|-f|-l|-s] [-E extensions] [-L languages]\n"
       << setw(strlen("Usage: ") + strlen(argv0)) << ' '
       << " [-S scanners] [-g] [-j numThreads] [-o] [-q] [-r|-a] path(s)\n\n"
       << "  -C : print configuration\n"
       << "  -c config : specify configuration file\n"
       << "  -d : print directory counters\n"
       << "  -e : print counters by file extension\n"
       << "  -f : print counters by file (default)\n"
       << "  -l : print counters by language\n"
       << "  -s : print counters by scanner type\n"
       << "  -E extensions : only scan files with extensions in the given comma-separated\n"
       << "                  list of extensions\n"
       << "  -L languages : only scan files for the languages in the given comma-separated\n"
       << "                 list of languages, which can include:\n";
  PrintLanguagesSupported(cerr);
  cerr << "  -S scanners : only enable scanners in the given comma-separated list of\n"
       << "                scanners, which can include:\n";
  PrintScannersSupported(cerr);
  cerr << "  -g : include generated files\n"
       << "  -j numThreads : specify the number of parsing threads"
       << " (default " << numThreads << ")\n"
       << "  -o : do not recurse into subdirectories\n"
       << "  -q : only show total\n"
       << "  -a : sort output by size, ascending\n"
       << "  -r : sort output by size, descending\n\n";
  exit(1);
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void
InitFromEnvironVariable(string & configPath,
                        Dwm::Mcloc::HowToSort & howToSort,
                        Dwm::Mcloc::SourceCollection::WhatToPrint & printBy,
                        int & numThreads,
                        set<string> & keepOnlyExtensions,
                        set<string> & keepOnlyLanguages,
                        set<string> & keepOnlyScanners,
                        bool & recurse, bool & quiet, bool & showGenerated,
                        bool & sortNormal, bool & reversed)
{
  using namespace  Dwm::Mcloc;
  
  EnvToArgs  envToArgs("MCLOC");

  //  Default configuration file.
  string  cfgPath(g_defaultConf);
  string  homeConfig(string(getenv("HOME")) + "/.mcloc");
  if (filesystem::exists(homeConfig)
      && filesystem::is_regular_file(homeConfig)) {
    //  We have a personal configuration file in our home directory.
    cfgPath = homeConfig;
  }
  configPath = cfgPath;
  
  //  Is the configuration file set in the environment?
  if (envToArgs.Exists('c')) {
    string  envConfig = envToArgs.Value('c');
    if (! envConfig.empty()) {
      if (filesystem::exists(envConfig)
          && filesystem::is_regular_file(envConfig)) {
        //  It's a file that exists, use it.
        configPath = envConfig;
      }
    }
  }
  
  //  Sorting set in the environment?
  howToSort = sortByName;
  if (envToArgs.Exists('a'))      { howToSort = sortBySize; }
  else if (envToArgs.Exists('r')) { howToSort = sortBySizeReversed; }

  //  Any settings from environment for what we'd like to print?
  printBy = SourceCollection::byFile;
  if (envToArgs.Exists('l')) { printBy = SourceCollection::byLanguage; }
  else if (envToArgs.Exists('d')) { printBy = SourceCollection::byDirectory; }
  else if (envToArgs.Exists('e')) { printBy = SourceCollection::byExtension; }
  else if (envToArgs.Exists('f')) { printBy = SourceCollection::byFile; }
  else if (envToArgs.Exists('s')) { printBy = SourceCollection::byScanner; }

  //  Number of threads set in environment?
  numThreads = thread::hardware_concurrency();
  if (numThreads > 2) {
    --numThreads;
  }
  if (envToArgs.Exists('j')) {
    numThreads = atoi(envToArgs.Value('j').c_str());
    if (! numThreads) {
      numThreads = 1;
    }
  }

  //  Filename extensions set in the environment?
  if (envToArgs.Exists('E')) {
    string  exts = envToArgs.Value('E');
    if (! exts.empty()) {
      KeepStrings(exts, keepOnlyExtensions, true);
    }
  }

  //  Languages set in environment?
  if (envToArgs.Exists('L')) {
    string  langs = envToArgs.Value('L');
    if (! langs.empty()) {
      KeepStrings(langs, keepOnlyLanguages, true);
    }
  }

  //  Scanners set in environment?
  if (envToArgs.Exists('S')) {
    string  scanners = envToArgs.Value('S');
    if (! scanners.empty()) {
      KeepStrings(scanners, keepOnlyScanners, true);
    }
  }
  //  Recurse into subdirectories?
  if (envToArgs.Exists('o')) { recurse = false; }
  //  Quiet output?
  if (envToArgs.Exists('q')) { quiet = true; }
  //  Count generated files?
  if (envToArgs.Exists('g')) { showGenerated = true; }
  //  Sort by size?
  if (envToArgs.Exists('a')) { sortNormal = true; }
  //  Reverse sort?
  if (envToArgs.Exists('r')) { reversed = true; }

  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  using Dwm::Mcloc::SourceCollection;

  string                         configPath;
  Dwm::Mcloc::HowToSort          howToSort;
  SourceCollection::WhatToPrint  printBy;
  int                            numThreads;
  set<string>                    keepOnlyExtensions;
  set<string>                    keepOnlyLanguages;
  set<string>                    keepOnlyScanners;
  bool                           recurse = true;
  bool                           quiet = false;
  bool                           showGenerated = false;
  bool                           sortBySize = false;
  bool                           reversed = false;
  
  InitFromEnvironVariable(configPath, howToSort, printBy, numThreads,
                          keepOnlyExtensions, keepOnlyLanguages,
                          keepOnlyScanners, recurse, quiet, showGenerated,
                          sortBySize, reversed);
  
  bool printConfig = false;

  
  int  optChar;
  while ((optChar = getopt(argc, argv, "CE:L:S:ac:dgj:loqefrst")) != -1) {
    switch (optChar) {
      case 'C': printConfig = true;                            break;
      case 'E': KeepStrings(optarg, keepOnlyExtensions);       break;
      case 'L': KeepStrings(optarg, keepOnlyLanguages, true);  break;
      case 'S': KeepStrings(optarg, keepOnlyScanners, true);   break;
      case 'c':
        if (filesystem::exists(optarg)
            && filesystem::is_regular_file(optarg)) {
          configPath = optarg;
        }
        break;
      case 'd': printBy = SourceCollection::byDirectory;     break;
      case 'e': printBy = SourceCollection::byExtension;     break;
      case 'f': printBy = SourceCollection::byFile;          break;
      case 'g': showGenerated = true;                        break;
      case 'j': numThreads = atoi(optarg);                   break;
      case 'l': printBy = SourceCollection::byLanguage;      break;
      case 'o': recurse = false;                             break;
      case 'q': quiet = true;                                break;
      case 'r': howToSort = Dwm::Mcloc::sortBySizeReversed;  break;
      case 'a': howToSort = Dwm::Mcloc::sortBySize;          break;
      case 's': printBy = SourceCollection::byScanner;       break;
      default:  Usage(argv[0], numThreads);                  break;
    }
  }

  Dwm::Mcloc::Config::Config  config;
  if (config.Parse(configPath)) {
    if (! keepOnlyExtensions.empty()) {
      config.KeepOnlyExtensions(keepOnlyExtensions);
    }
    if (! keepOnlyLanguages.empty()) {
      config.KeepOnlyLanguages(keepOnlyLanguages);
    }
    if (! keepOnlyScanners.empty()) {
      config.KeepOnlyScanners(keepOnlyScanners);
    }
    
    if (printConfig) {
      cout << configPath;
      if (! keepOnlyExtensions.empty()) {
        cout << ", modified by -E";
      }
      if (! keepOnlyLanguages.empty()) {
        cout << ", modified by -L";
      }
      if (! keepOnlyScanners.empty()) {
        cout << ", modified by -S";
      }
      cout << ":\n\n"
           << config;
      return 0;
    }
    
    if (optind >= argc) {
      Usage(argv[0], numThreads);
    }

    Dwm::Mcloc::CodeUtils::Initialize(config);

    SourceCollection  sc;
    for (int i = optind; i < argc; ++i) {
      if (filesystem::exists(argv[i])) {
        if (filesystem::is_directory(argv[i])) {
          sc.AddDirectory(argv[i], recurse);
        }
        else if (filesystem::is_regular_file(argv[i])) {
          sc.AddFile(argv[i]);
        }
      }
    }
    sc.Parse(numThreads);
    cout.imbue(std::locale(""));
    
    sc.Print(cout, printBy, showGenerated, howToSort, quiet);
    
    return 0;
  }
  else {
    cerr << "Failed to parse configuration in '" << configPath << "'\n";
    Usage(argv[0], numThreads);
  }
  return 1;
}
