//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcBlockServerConfig.hh 12091 2022-11-29 05:05:27Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockServerConfig.hh
//!  \brief Dwm::McBlock::ServerConfig class definition
//---------------------------------------------------------------------------

#ifndef _DWMMCBLOCKSERVERCONFIG_HH_
#define _DWMMCBLOCKSERVERCONFIG_HH_

#include <string>
#include <nlohmann/json.hpp>

#include "DwmIpv4Address.hh"
#include "DwmCfg2Json.hh"

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  Encapsulates the mcblockd server configuration.
    //------------------------------------------------------------------------
    class ServerConfig
    {
    public:
      //----------------------------------------------------------------------
      //!  Constructor
      //----------------------------------------------------------------------
      ServerConfig(const Cfg2Json & config);

      //----------------------------------------------------------------------
      //!  Returns the configured PF device path.
      //----------------------------------------------------------------------
      const std::string & Device() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the configured PF device path.
      //----------------------------------------------------------------------
      const std::string & Device(const std::string & path);

      //----------------------------------------------------------------------
      //!  Returns the TCP address to which the server will bind.
      //----------------------------------------------------------------------
      const Ipv4Address & TCPAddr() const;

      //----------------------------------------------------------------------
      //!  Sets and returns the TCP address to which the server will bind.
      //----------------------------------------------------------------------
      const Ipv4Address & TCPAddr(const Ipv4Address & addr);
      
      //----------------------------------------------------------------------
      //!  Returns the TCP port number.
      //----------------------------------------------------------------------
      uint16_t TCPPort() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the TCP port number.
      //----------------------------------------------------------------------
      uint16_t TCPPort(uint16_t tcpPort);
        
      //----------------------------------------------------------------------
      //!  Returns the multicast address that will be used for status
      //!  messages.
      //----------------------------------------------------------------------
      const Ipv4Address & MulticastAddr() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the multicast address that will be used for
      //!  status messages.
      //----------------------------------------------------------------------
      const Ipv4Address & MulticastAddr(const Ipv4Address & addr);
        
      //----------------------------------------------------------------------
      //!  Returns the UDP port number used for multicast status messages.
      //----------------------------------------------------------------------
      uint16_t MulticastPort() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the UDP port number used for multicast status
      //!  messages.
      //----------------------------------------------------------------------
      uint16_t MulticastPort(uint16_t port);

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      const std::string & KeyDirectory() const;
      
      //----------------------------------------------------------------------
      //!  Return the dwmrdap server hostname.
      //----------------------------------------------------------------------
      const std::string & DwmRDAPServer() const;

      //----------------------------------------------------------------------
      //!  Returns the directory where blocked database files are stored.
      //----------------------------------------------------------------------
      const std::string & DatabaseDirectory() const;
      
      //----------------------------------------------------------------------
      //!  Returns the server configuration as a JSON value.
      //----------------------------------------------------------------------
      nlohmann::json Json() const;
        
      //----------------------------------------------------------------------
      //!  Sets the server configuration from a JSON value.
      //----------------------------------------------------------------------
      nlohmann::json Json(const nlohmann::json & value);
        
      //----------------------------------------------------------------------
      //!  == operator
      //----------------------------------------------------------------------
      bool operator == (const ServerConfig & config) const;
      
    private:
      std::string  _pfdevpath;
      Ipv4Address  _tcpAddr;
      uint16_t     _tcpPort;
      Ipv4Address  _mcastAddr;
      uint16_t     _mcastPort;
      std::string  _keyDirectory;
      std::string  _dwmRDAPServer;
      std::string  _databaseDirectory;
    };
    
  }  // namespace McBlock

}  // namespace Dwm

#endif  // _DWMMCBLOCKSERVERCONFIG_HH_
