//===========================================================================
// @(#) $Name$
// @(#) $Id: DwmMcBlockDbEntry.hh 12091 2022-11-29 05:05:27Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2015
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockDbEntry.hh
//!  \brief Dwm::McBlock::DbEntry class definition
//---------------------------------------------------------------------------

#ifndef _DWMMCBLOCKDBENTRY_HH_
#define _DWMMCBLOCKDBENTRY_HH_


#include <tuple>
#include <vector>

#include "DwmBZ2IO.hh"
#include "DwmGZIO.hh"
#include "DwmIO.hh"
#include "DwmIpv4Prefix.hh"
#include "DwmTimeInterval.hh"
#include "DwmDescriptorIOCapable.hh"
#include "DwmFileIOCapable.hh"
#include "DwmDescriptorIOCapable.hh"
#include "DwmBZ2IOCapable.hh"
#include "DwmGZIOCapable.hh"
#include "DwmStreamedLengthCapable.hh"
#include "DwmCountryCodes.hh"
#include "DwmMcBlockIpv4Routes.hh"

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  Encapsulate information for a single entry stored in a McBlock::Db.
    //!  IPv4 prefix, time interval, country and registry.
    //------------------------------------------------------------------------
    class DbEntry
      : public StreamIOCapable, public GZIOCapable,
        public FileIOCapable, public DescriptorIOCapable,
        public BZ2IOCapable, public StreamedLengthCapable
    {
    public:
      //----------------------------------------------------------------------
      //!  Default constructor
      //----------------------------------------------------------------------
      DbEntry();

      //----------------------------------------------------------------------
      //!  Construct from a prefix.
      //----------------------------------------------------------------------
      DbEntry(const Ipv4Prefix & prefix);
      
      //----------------------------------------------------------------------
      //!  Construct from a given @c prefix, @c interval, @c registry and
      //!  country.
      //----------------------------------------------------------------------
      DbEntry(const Ipv4Prefix & prefix, const TimeInterval & interval,
              const std::string & registry, const std::string & country);
      
      //----------------------------------------------------------------------
      //!  Returns the contained prefix.
      //----------------------------------------------------------------------
      inline const Ipv4Prefix & Prefix() const
      {
        return std::get<0>(_data);
      }

      //----------------------------------------------------------------------
      //!  Sets and returns the contained prefix.
      //----------------------------------------------------------------------
      inline const Ipv4Prefix & Prefix(const Ipv4Prefix & prefix)
      {
        std::get<0>(_data) = prefix;
        return std::get<0>(_data);
      }

      //----------------------------------------------------------------------
      //!  Returns a const reference to the contained interval.
      //----------------------------------------------------------------------
      inline const TimeInterval & Interval() const
      {
        return std::get<1>(_data);
      }
      
      //----------------------------------------------------------------------
      //!  Returns a mutable reference to the contained interval.
      //----------------------------------------------------------------------
      inline TimeInterval & Interval()
      {
        return std::get<1>(_data);
      }

      //----------------------------------------------------------------------
      //!  Sets and returns the contained interval.
      //----------------------------------------------------------------------
      inline const TimeInterval & Interval(const TimeInterval & interval)
      {
        std::get<1>(_data) = interval;
        return std::get<1>(_data);
      }

      //----------------------------------------------------------------------
      //!  Returns the contained registry.
      //----------------------------------------------------------------------
      inline const std::string & Registry() const
      {
        return std::get<2>(_data);
      }

      //----------------------------------------------------------------------
      //!  Sets and returns the contained registry.
      //----------------------------------------------------------------------
      inline const std::string & Registry(const std::string & registry)
      {
        std::get<2>(_data) = registry;
        return std::get<2>(_data);
      }

      //----------------------------------------------------------------------
      //!  Returns the contained country.
      //----------------------------------------------------------------------
      inline const std::string & Country() const
      {
        return std::get<3>(_data);
      }

      //----------------------------------------------------------------------
      //!  Sets and returns the contained country.
      //----------------------------------------------------------------------
      inline const std::string & Country(const std::string & country)
      {
        std::get<3>(_data) = country;
        return std::get<3>(_data);
      }

      //----------------------------------------------------------------------
      //!  Returns the default registry for the given @c prefix.
      //----------------------------------------------------------------------
      static std::string GetDefaultRegistry(const Ipv4Prefix & prefix);
      
      //----------------------------------------------------------------------
      //!  Returns true if the entry is active (i.e. the contained interval
      //!  spans the current time).
      //----------------------------------------------------------------------
      bool IsActive() const;

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      std::string DurationString() const;
      
      //----------------------------------------------------------------------
      //!  Reads the entry from an istream.  Returns the istream.
      //----------------------------------------------------------------------
      std::istream & Read(std::istream & is) override
      {
        if (IO::Read(is, _data)) {
          if (std::get<2>(_data).empty()) {
            std::get<2>(_data) = GetDefaultRegistry(std::get<0>(_data));
          }
        }
        return is;
      }

      //----------------------------------------------------------------------
      //!  Writes the entry to an ostream.  Returns the ostream.
      //----------------------------------------------------------------------
      std::ostream & Write(std::ostream & os) const override
      {
        return IO::Write(os, _data);
      }

      //----------------------------------------------------------------------
      //!  Reads the entry from a file descriptor.  Returns the number of
      //!  bytes read on success, -1 on failure.
      //----------------------------------------------------------------------
      ssize_t Read(int fd) override
      {
        ssize_t  rc = IO::Read(fd, _data);
        if (rc > 0) {
          if (std::get<2>(_data).empty()) {
            std::get<2>(_data) = GetDefaultRegistry(std::get<0>(_data));
          }
        }
        return rc;
      }
      
      //----------------------------------------------------------------------
      //!  Writes the entry to a file descriptor.  Returns the number of
      //!  bytes written on success, -1 on failure.
      //----------------------------------------------------------------------
      ssize_t Write(int fd) const override
      {
        return IO::Write(fd, _data);
      }

      //----------------------------------------------------------------------
      //!  Reads the entry from a FILE.  Returns 1 on success, 0 on failure
      //!  (fread() semantics).
      //----------------------------------------------------------------------
      size_t Read(FILE *f) override
      {
        size_t  rc = IO::Read(f, _data);
        if (rc) {
          if (std::get<2>(_data).empty()) {
            std::get<2>(_data) = GetDefaultRegistry(std::get<0>(_data));
          }
        }
        return rc;
      }

      //----------------------------------------------------------------------
      //!  Writes the entry to a FILE.  Returns 1 on success, 0 on failure
      //!  (fwrite() semantics).
      //----------------------------------------------------------------------
      size_t Write(FILE *f) const override
      {
        return IO::Write(f, _data);
      }

      //----------------------------------------------------------------------
      //!  Reads the entry from a gzFile.  Returns the number of bytes read.
      //----------------------------------------------------------------------
      int Read(gzFile gzf) override
      {
        return GZIO::Read(gzf, _data);
      }

      //----------------------------------------------------------------------
      //!  Writes the entry to a gzFile.  Returns the number of bytes written.
      //----------------------------------------------------------------------
      int Write(gzFile gzf) const override
      {
        return GZIO::Write(gzf, _data);
      }

      //----------------------------------------------------------------------
      //!  Reads the entry from a BZFILE.  Returns the number of bytes read.
      //----------------------------------------------------------------------
      int BZRead(BZFILE *bzf) override
      {
        return BZ2IO::BZRead(bzf, _data);
      }

      //----------------------------------------------------------------------
      //!  Writes the entry to a BZFILE.  Returns the number of bytes written.
      //----------------------------------------------------------------------
      int BZWrite(BZFILE *bzf) const override
      {
        return BZ2IO::BZWrite(bzf, _data);
      }

      //----------------------------------------------------------------------
      //!  Returns the number of bytes that would be written if one of the
      //!  Write() members were called.
      //----------------------------------------------------------------------
      uint64_t StreamedLength() const override
      {
        return IO::StreamedLength(_data);
      }

      //----------------------------------------------------------------------
      //!  operator ==
      //----------------------------------------------------------------------
      bool operator == (const DbEntry & dbe) const
      {
        return (_data == dbe._data);
      }

      //----------------------------------------------------------------------
      //!  Writes an entry to an ostream in human-readable form.
      //----------------------------------------------------------------------
      friend std::ostream &
      operator << (std::ostream & os, const DbEntry & dbe);

      //----------------------------------------------------------------------
      //!  Returns a full country name for the entry.
      //----------------------------------------------------------------------
      std::string GetCountryName() const;
      
      //----------------------------------------------------------------------
      //!  Returns a full country name for the given @c countryCode.
      //----------------------------------------------------------------------
      static std::string GetCountryName(const std::string & countryCode);
      
    private:
      static Ipv4Routes<std::string>  _defaultRegistryRoutes;
      static CountryCodes             _countryCodes;
      
      std::tuple<Ipv4Prefix,    // prefix
                 TimeInterval,  // time interval
                 std::string,   // registry
                 std::string    // country
                 > _data;

      static bool LoadDefaultRegistryRoutes();
    };
    
  }  // namespace McBlock

}  // namespace Dwm

#endif  // _DWMMCBLOCKDBENTRY_HH_
