//===========================================================================
// @(#) $Name:$
// @(#) $Id: mcblocklog.cc 12114 2022-12-05 04:02:55Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file mcblocklogparser.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <arpa/inet.h>
  #include <netdb.h>
}

#include <fstream>
#include <map>

#include "DwmIpv4Prefix.hh"
#include "DwmOptArgs.hh"
#include "DwmSignal.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcBlockLogRegexMatcher.hh"
#include "DwmMcBlockRequestMessage.hh"
#include "DwmMcBlockResponseMessage.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcblock/tags/mcblock-0.3.3/apps/mcblocklog/mcblocklog.cc 12114 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool ConnectToMcBlock(const string & host, Credence::Peer & peer)
{
  bool  rc = false;
  struct hostent  *hostEntry = gethostbyname(host.c_str());
  if (! hostEntry) {
    Syslog(LOG_ERR, "Host '%s' not found", host.c_str());
    exit(1);
  }
  if (peer.Connect(host, 1001)) {
    rc = true;
  }
  else {
    Syslog(LOG_ERR, "Failed to connect to '%s'", host.c_str());
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool RequestResponse(Credence::Peer & peer,
                     const McBlock::RequestMessage & request,
                     McBlock::ResponseMessage & response)
{
  bool  rc = false;
  if (request.Write(peer)) {
    if (response.Read(peer)) {
      rc = true;
    }
    else {
      Syslog(LOG_ERR, "Failed to read server response");
    }
  }
  else {
    Syslog(LOG_ERR, "Failed to write server request");
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void LogHit(const string & mcblockServer, const string & tableName,
                   const Ipv4Address & addr)
{
  Credence::Peer peer;
  
  if (ConnectToMcBlock(mcblockServer, peer)) {
    Credence::KeyStash   keyStash("/usr/local/etc/mcblocklog");
    Credence::KnownKeys  knownKeys("/usr/local/etc/mcblocklog");
    if (peer.Authenticate(keyStash, knownKeys)) {
      Dwm::TimeValue64  now(true);
      McBlock::ResponseMessage  resp;
      McBlock::RequestMessage   req((McBlock::LogHitReq(tableName, addr,
                                                        now)));
      RequestResponse(peer, req, resp);
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  OptArgs  optargs;
  optargs.AddOptArg("r:", "regexFile", true, "",
                    "file containing regular expressions");
  optargs.AddOptArg("t:", "tablename", true, "",
                    "pf table name");
  optargs.AddOptArg("h:", "mcblockHost", true, "",
                    "mcblock host");
  int  nextArg = optargs.Parse(argc, argv);

  SysLogger::Open("mcblocklog", LOG_PID, "local0");

  McBlock::LogRegexMatcher  matcher(optargs.Get<string>('r'));
  map<Ipv4Address,uint32_t>  matches;
  
  Signal SigPipe(SIGPIPE);
  SigPipe.Block();
  Signal SigTerm(SIGTERM);
  SigTerm.Block();

  Syslog(LOG_INFO, "Started");
  
  string  line;

  for (;;) {
    if (SigPipe.IsPending() || SigTerm.IsPending()) {
      Syslog(LOG_INFO, "Exiting from signal");
      break;
    }
    if (getline(cin, line, '\n')) {
      pair<bool,Ipv4Address>  match = matcher.Matches(line);
      if (match.first) {
        LogHit(optargs.Get<string>('h'), optargs.Get<string>('t'),
               match.second);
      }
    }
    else {
      Syslog(LOG_INFO, "Exiting from failed input read");
      break;
    }
  }
  return 0;
}
