//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcBlockResponder.cc 12115 2022-12-06 03:41:24Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017, 2022
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockResponder.cc
//!  \brief Dwm::McBlock::Responder class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/ioctl.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <arpa/inet.h>
  #include <netdb.h>
#ifndef __APPLE__
  #include <pthread_np.h>
#endif
}

#include "DwmSysLogger.hh"
#include "DwmSvnTag.hh"
#include "DwmMcBlockPortability.hh"
#include "DwmMcBlockRequestMessage.hh"
#include "DwmMcBlockResponseMessage.hh"
#include "DwmMcBlockServer.hh"
#include "DwmMcBlockTcpDrop.hh"

static const Dwm::SvnTag  svntag("$DwmPath: dwm/mcplex/mcblock/tags/mcblock-0.3.3/apps/mcblockd/DwmMcBlockResponder.cc 12115 $");

using namespace std;

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Responder::Responder(boost::asio::ip::tcp::socket && s, Server & server)
        : _peer(), _server(server), _running(false)
    {
      boost::system::error_code  ec;
      s.native_non_blocking(false, ec);
      _peer.SetKeyExchangeTimeout(chrono::milliseconds(2000));
      _peer.SetIdExchangeTimeout(chrono::milliseconds(2000));
      if (_peer.Accept(std::move(s))) {
        _run = true;
        _thread = std::thread(&Responder::Run, this);
        _running.store(true);
      }
      else {
        _peer.Disconnect();
        Syslog(LOG_ERR, "Failed to accept peer");
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::Stop()
    {
      _run = false;
      return Join();
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::Join()
    {
      bool  rc = false;
      if (! _running.load()) {
        if (_thread.joinable()) {
          _thread.join();
          rc = true;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidGetRequest(const nlohmann::json & request)
    {
      return ((request.find("type") != request.end())
              && (request["type"].is_string())
              && (request["type"].get<string>() == "get")
              && (request.find("table") != request.end())
              && (request["table"].is_string())
              && (request.find("ipv4addr") != request.end())
              && (request["ipv4addr"].is_string()));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidEditRequest(const nlohmann::json & request)
    {
      return ((request.find("type") != request.end())
              && request["type"].is_string()
              && (request["type"].get<string>() == "edit")
              && (request.find("table") != request.end())
              && request["table"].is_string()
              && (request.find("prefix") != request.end())
              && request["prefix"].is_string()
              && (request.find("countryCode") != request.end())
              && request["countryCode"].is_string()
              && (request["countryCode"].get<string>().size() == 2)
              && (request.find("daysRemaining") != request.end())
              && request["daysRemaining"].is_number_unsigned());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidActivateRequest(const nlohmann::json & request)
    {
      return ((request.find("type") != request.end())
              && (request["type"].is_string())
              && (request["type"].get<string>() == "activate")
              && (request.find("table") != request.end())
              && (request["table"].is_string())
              && (request.find("prefixes") != request.end())
              && (request["prefixes"].is_array())
              && (request["prefixes"].size() > 0)
              && (request["prefixes"][0].is_string()));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidDeactivateRequest(const nlohmann::json & request)
    {
      return ((request.find("type") != request.end())
              && (request["type"].is_string())
              && (request["type"].get<string>() == "deactivate")
              && (request.find("table") != request.end())
              && (request["table"].is_string())
              && (request.find("prefixes") != request.end())
              && (request["prefixes"].is_array())
              && (request["prefixes"].size() > 0)
              && (request["prefixes"][0].is_string()));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidGetActiveRequest(const nlohmann::json & request)
    {
      return ((! request.empty())
              && (request.find("type") != request.end())
              && (request["type"].is_string())
              && (request["type"].get<string>() == "getActive")
              && (request.find("table") != request.end())
              && (request["table"].is_string()));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidSearchRequest(const nlohmann::json & request)
    {
      return ((! request.empty())
              && (request.find("type") != request.end())
              && (request["type"].is_string())
              && (request["type"].get<string>() == "search")
              && (request.find("ipv4addr") != request.end())
              && (request["ipv4addr"].is_string()));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidGetAddRulesRequest(const nlohmann::json & request)
    {
      return ((! request.empty())
              && (request.find("type") != request.end())
              && (request["type"].is_string())
              && (request["type"].get<string>() == "getAddRules")
              && (request.find("table") != request.end())
              && (request["table"].is_string()));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidLogHitRequest(const nlohmann::json & request)
    {
      return ((! request.empty())
              && (request.find("type") != request.end())
              && (request["type"].is_string())
              && (request["type"].get<string>() == "logHit")
              && (request.find("table") != request.end())
              && (request["table"].is_string())
              && (request.find("ipv4addr") != request.end())
              && (request["ipv4addr"].is_string())
              && (request.find("logTime") != request.end())
              && (request["logTime"].is_number_unsigned()));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidPingRequest(const nlohmann::json & request)
    {
      return ((! request.empty())
              && (request.find("type") != request.end())
              && (request["type"].is_string())
              && (request["type"].get<string>() == "ping"));
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RDAP::FetchedEntryMap
    Responder::GetRDAPEntries(const vector<Ipv4Address> & addrs)
    {
      RDAP::FetchedEntryMap  rc;
      Credence::KeyStash     keyStash(_server.Config().KeyDirectory());
      Credence::KnownKeys    knownKeys(_server.Config().KeyDirectory());
      RDAP::Fetcher          fetcher;
      if (fetcher.OpenSession(_server.Config().DwmRDAPServer(),
                              keyStash, knownKeys)) {
        rc = fetcher.GetEntries(addrs);
        fetcher.CloseSession();
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::HandleActivateRequest(const nlohmann::json & request)
    {
      Syslog(LOG_DEBUG, "request %s", request.dump().c_str());
      nlohmann::json       rc;
      vector<Ipv4Prefix>   prefixes;
      vector<Ipv4Address>  addrs;
      
      Pf::Table  &&table =
        _server.GetPfDevice().GetTable("", request["table"].get<string>());
      if (table.Name() == request["table"].get<string>()) {
        for (uint32_t i = 0; i < request["prefixes"].size(); ++i) {
          Ipv4Prefix  prefix(request["prefixes"][i].get<string>());
          prefixes.push_back(prefix);
          if (prefix.MaskLength() == 32) {
            addrs.push_back(prefix.FirstAddress());
          }
        }
        RDAP::FetchedEntryMap &&rdapEntries = GetRDAPEntries(addrs);
        for (auto & rde : rdapEntries) {
          Syslog(LOG_INFO, "RDAP country %s prefix %s",
                 rde.second.Country().c_str(),
                 rde.second.Prefix().ToShortString().c_str());
        }
        for (auto ai = 0; ai < prefixes.size(); ++ai) {
          Ipv4Prefix  match(prefixes[ai]);
          if (! table.Contains(prefixes[ai], match)) {
            if (table.Add(match)) {
              KillSourceState(match);
              TcpDropPrefix(match);
              AddToDatabase(table.Name(), prefixes[ai]);
              rc["table"] = request["table"];
              rc["prefixes"][ai]["requested"] = request["prefixes"][ai];
              rc["prefixes"][ai]["added"] = match.ToShortString();
            }
            else {
              Syslog(LOG_ERR, "Failed to add %s to %s",
                     prefixes[ai].ToString().c_str(), table.Name().c_str());
            }
          }
          else {
            rc["table"] = request["table"];
            rc["prefixes"][ai]["requested"] = request["prefixes"][ai];
            rc["prefixes"][ai]["matched"] = match.ToShortString();
          }
        }
      }
      else {
        Syslog(LOG_ERR, "Table '%s' not found",
               request["table"].get<string>().c_str());
      }
      
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::HandleDeactivateRequest(const nlohmann::json & request)
    {
      nlohmann::json    rc;
      Pf::Table       &&table =
        _server.GetPfDevice().GetTable("", request["table"].get<string>());
      if (table.Name() == request["table"].get<string>()) {
        for (uint32_t i = 0; i < request["prefixes"].size(); ++i) {
          Ipv4Prefix  prefix(request["prefixes"][i].get<string>());
          Ipv4Prefix  match(prefix);
          if (table.Contains(prefix, match)) {
            if (match == prefix) {
              if (table.Remove(prefix)) {
                RemoveFromDatabase(table.Name(), prefix);
                rc["table"] = request["table"];
                rc["prefixes"][i]["deactivated"] = request["prefixes"][i];
              }
            }
            else {
              rc["table"] = request["table"];
              rc["prefixes"][i]["not_deactivated"] = match.ToShortString();
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::HandleGetActiveRequest(const nlohmann::json & request)
    {
      nlohmann::json  rc;
      string          tableName = request["table"].get<string>();
      auto  dbi = _server.GetDatabases().begin();
      if (! tableName.empty()) {
        dbi = _server.GetDatabases().find(tableName);
      }
      
      for ( ; dbi != _server.GetDatabases().end(); ++dbi) {
        vector<DbEntry>  dbEntries;
        dbi->second.GetAllActive(dbEntries);
        if (! dbEntries.empty()) {
          TimeValue  now(true);
          uint32_t  i = 0;
          for (const auto & entry : dbEntries) {
            rc[dbi->first]["prefixes"][i]["prefix"] =
              entry.Prefix().ToShortString();
            rc[dbi->first]["prefixes"][i]["country"] = entry.Country();
            rc[dbi->first]["prefixes"][i]["countryName"] =
              entry.GetCountryName(entry.Country());
            TimeValue  timeLeft(entry.Interval().End());
            timeLeft -= now;
            uint32_t  daysLeft = (timeLeft.Secs() / (24 * 60 * 60));
            rc[dbi->first]["prefixes"][i]["daysRemaining"] = daysLeft;
            ++i;
          }
        }
        if (! tableName.empty()) {
          break;
        }
      }
      
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::HandleGetRequest(const nlohmann::json & request)
    {
      nlohmann::json  rc;
      Ipv4Address     addr(request["ipv4addr"].get<string>());
      auto dbi = _server.GetDatabases().find(request["table"].get<string>());
      if (dbi != _server.GetDatabases().end()) {
        pair<Ipv4Prefix,DbEntry>  match;
        if (dbi->second.FindLongest(addr, match)) {
          rc["table"] = dbi->first;
          rc["prefix"] = match.second.Prefix().ToShortString();
          rc["countryCode"] = match.second.Country();
          TimeValue  now(true);
          TimeValue  timeLeft(match.second.Interval().End());
          timeLeft -= now;
          uint32_t  daysLeft = (timeLeft.Secs() / (24 * 60 * 60));
          rc["daysRemaining"] = daysLeft;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::HandleEditRequest(const nlohmann::json & request)
    {
      nlohmann::json  rc;
      Ipv4Prefix      pfx(request["prefix"].get<string>());
      auto dbi = _server.GetDatabases().find(request["table"].get<string>());
      if (dbi != _server.GetDatabases().end()) {
        DbEntry  match;
        if (dbi->second.Find(pfx, match)) {
          TimeValue  now(true);
          TimeValue  endTime(now);
          endTime.Set(endTime.Secs()
                      + (request["daysRemaining"].get<uint64_t>() * 24 * 60 * 60), 0);
          TimeInterval  interval(match.Interval().Start(), endTime);
          match.Interval(interval);
          match.Country(request["countryCode"].get<string>());
          dbi->second[pfx] = match;
          dbi->second.Save("");
          rc["edited"] = true;
          rc["prefix"] = pfx.ToShortString();
          rc["table"] = dbi->first;
          rc["countryCode"] = match.Country();
          TimeValue  timeLeft(match.Interval().End());
          timeLeft -= now;
          uint32_t  daysLeft = (timeLeft.Secs() / (24 * 60 * 60));
          rc["daysRemaining"] = daysLeft;
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::HandleSearchRequest(const nlohmann::json & request)
    {
      nlohmann::json  rc;
      Ipv4Address     addr(request["ipv4addr"].get<string>());
      TimeValue       now(true);
      for (const auto & db : _server.GetDatabases()) {
        pair<Ipv4Prefix,DbEntry>  match;
        if (db.second.FindLongest(addr, match)) {
          if (match.second.IsActive()) {
            rc[db.first]["prefix"] = match.second.Prefix().ToShortString();
            rc[db.first]["country"] = match.second.Country();
            rc[db.first]["countryName"] =
              DbEntry::GetCountryName(match.second.Country());
            TimeValue  timeLeft(match.second.Interval().End());
            timeLeft -= now;
            uint32_t  daysLeft = (timeLeft.Secs() / (24 * 60 * 60));
            rc[db.first]["daysRemaining"] = daysLeft;
          }
        }
      }
      return rc;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::HandleGetAddRulesRequest(const nlohmann::json & request)
    {
      nlohmann::json  rc;
      auto  ari =
        _server.GetAddRules().RulesForTable(request["table"].get<string>());
      if (ari != _server.GetAddRules().RulesForTables().end()) {
        rc = ari->second.Json();
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::HandlePingRequest(const nlohmann::json & request)
    {
      nlohmann::json  rc;
      rc["alive"] = true;
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Responder::GetRDAPEntry(const Ipv4Address & addr,
                                 Ipv4Prefix & prefix, string & country)
    {
      RDAP::Fetcher                  fetcher;
      pair<bool,RDAP::FetchedEntry>  rdapResp;  rdapResp.first = false;
      Credence::KeyStash             keyStash(_server.Config().KeyDirectory());
      Credence::KnownKeys            knownKeys(_server.Config().KeyDirectory());
      
      if (fetcher.OpenSession(_server.Config().DwmRDAPServer(),
                              keyStash, knownKeys)) {
        rdapResp = fetcher.GetEntry(addr);
        fetcher.CloseSession();
      }
      if (rdapResp.first) {
        prefix = rdapResp.second.Prefix();
        country = rdapResp.second.Country();
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Responder::AddToDatabase(const string & tableName,
                                  const Ipv4Prefix & prefix,
                                  uint32_t days, const string & country)
    {
      auto  dbi = _server.GetDatabases().find(tableName);
      if (dbi != _server.GetDatabases().end()) {
        TimeValue  now(true);
        TimeValue  endTime(now);
        endTime.Set(endTime.Secs() + (days * 24 * 60 * 60), 0);
        TimeInterval  ti(now, endTime);
        DbEntry  dbEntry(prefix, ti, "", country);
        dbi->second.AddEntry(dbEntry);
        dbi->second.Save("");
      }
      else {
        Syslog(LOG_ERR, "Database not found for table %s", tableName.c_str());
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Responder::AddToDatabase(const string & tableName,
                                  const Ipv4Prefix & prefix)
    {
      auto  dbi = _server.GetDatabases().find(tableName);
      if (dbi != _server.GetDatabases().end()) {
        DbEntry  dbEntry;
        if (dbi->second.Find(prefix, dbEntry)) {
          dbi->second.DeleteEntry(prefix);
        }
        DbEntry  newDbEntry(prefix);
        dbi->second.AddEntry(newDbEntry);
        dbi->second.Save("");
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Responder::RemoveFromDatabase(const string & tableName,
                                       const Ipv4Prefix & prefix)
    {
      auto  dbi = _server.GetDatabases().find(tableName);
      if (dbi != _server.GetDatabases().end()) {
        if (dbi->second.DeleteEntry(prefix)) {
          dbi->second.Save("");
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::KillSourceState(const Ipv4Prefix & prefix)
    {
      struct pfioc_state_kill psk;
      memset(&psk, 0, sizeof(psk));
      
      psk.psk_af = AF_INET;
      psk.psk_src.addr.v.a.addr.v4.s_addr = prefix.Network().Raw();
      psk.psk_src.addr.v.a.mask.v4.s_addr = prefix.Netmask().Raw();
      
      return _server.GetPfDevice().Ioctl(DIOCKILLSTATES, &psk);
    }

    //------------------------------------------------------------------------
    //!  This is gnarly.  How can I refactor it to make it easier to maintain?
    //------------------------------------------------------------------------
    nlohmann::json
    Responder::HandleLogHitRequest(const nlohmann::json & request)
    {
      nlohmann::json  rc;
      Ipv4Address     addr(request["ipv4addr"].get<string>());
      Pf::Table  &&pftable =
        _server.GetPfDevice().GetTable("", request["table"].get<string>());
      if (pftable.Name() == request["table"].get<string>()) {
        Ipv4Prefix  match;
        if (pftable.Contains(addr, match)) {
          // already blocked
          rc["requested"] = request["ipv4addr"];
          rc["matched"] = match.ToShortString();
          return rc;
        }
        else {
          //  not yet blocked.
          Ipv4Prefix  prefix(addr, 24);
          string      country("??");
          //  Try to get prefix and country from dwmrdapd.
          GetRDAPEntry(addr, prefix, country);

          AddRule  addRule;
          //  Get the rules for the table
          auto  ari =
            _server.GetAddRules().RulesForTable(request["table"].get<string>());
          if (ari != _server.GetAddRules().RulesForTables().end()) {
            //  And the rule for the country
            if (ari->second.FindRuleForCountry(country, addRule)) {
              if (addRule.WidestMask() > prefix.MaskLength()) {
                //  apppy widest mask from rule
                prefix = Ipv4Prefix(addr, addRule.WidestMask());
              }
            }
            else {
              //  no rule for country!
              prefix = Ipv4Prefix(addr, 24);
            }
            if (addRule.LogThresh() <= 1) {
              //  special case, no tracking necessary
              if (pftable.Add(prefix)) {
                KillSourceState(prefix);
                TcpDropPrefix(prefix);
                AddToDatabase(pftable.Name(), prefix, addRule.Days(), country);
                Syslog(LOG_INFO, "Added %s (%s) to %s for %u days",
                       prefix.ToShortString().c_str(), country.c_str(),
                       pftable.Name().c_str(), addRule.Days());
                rc["requested"] = request["ipv4addr"];
                rc["added"] = prefix.ToShortString();
              }
            }
            else {
              TimeValue64  logtime(request["logTime"].get<uint64_t>(), 0);
              auto  leti = _server.GetLogEntryTrackers().find(request["table"].get<string>());
              if (leti != _server.GetLogEntryTrackers().end()) {
                leti->second.Add(prefix, country, logtime);
                if (leti->second.HitThreshold(prefix, ari->second)) {
                  //  Above configured hit threshold
                  if (pftable.Add(prefix)) {
                    KillSourceState(prefix);
                    TcpDropPrefix(prefix);
                    AddToDatabase(pftable.Name(), prefix, addRule.Days(),
                                  country);
                    Syslog(LOG_INFO, "Added %s (%s) to %s for %u days",
                           prefix.ToShortString().c_str(), country.c_str(),
                           pftable.Name().c_str(), addRule.Days());
                    rc["requested"] = request["ipv4addr"];
                    rc["added"] = prefix.ToShortString();
                  }
                  leti->second.Remove(prefix);
                }
                else {
                  //  Not above threshold yet
                  Syslog(LOG_INFO, "Pending %s (%s) for %s, %u/%u",
                         prefix.ToShortString().c_str(), country.c_str(),
                         pftable.Name().c_str(),
                         leti->second.CurrentHitLevel(prefix, ari->second),
                         addRule.LogThresh());
                  rc["requested"] = request["ipv4addr"];
                  rc["pending"] = prefix.ToShortString();
                }
              }
              else {
                //  No log tracker found, add it
                TableLogEntryTracker  & letRef =
                  _server.GetLogEntryTrackers()[request["table"].get<string>()];
                letRef.Add(prefix, country, logtime);
                Syslog(LOG_INFO, "Pending %s (%s) for %s, %u/%u",
                       prefix.ToShortString().c_str(), country.c_str(),
                       pftable.Name().c_str(),
                       letRef.CurrentHitLevel(prefix, ari->second),
                       addRule.LogThresh());
                rc["requested"] = request["ipv4addr"];
                rc["pending"] = prefix.ToShortString();
              }
            }
          }
          else {
            //  No rules for table!
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json Responder::HandleRequest(const nlohmann::json & request)
    {
      Syslog(LOG_DEBUG, "Handling request '%s'", request.dump().c_str());
      
      if (IsValidActivateRequest(request)) {
        return HandleActivateRequest(request);
      }
      else if (IsValidDeactivateRequest(request)) {
        return HandleDeactivateRequest(request);
      }
      else if (IsValidSearchRequest(request)) {
        return HandleSearchRequest(request);
      }
      else if (IsValidGetActiveRequest(request)) {
        return HandleGetActiveRequest(request);
      }
      else if (IsValidGetAddRulesRequest(request)) {
        return HandleGetAddRulesRequest(request);
      }
      else if (IsValidLogHitRequest(request)) {
        return HandleLogHitRequest(request);
      }
      else if (IsValidGetRequest(request)) {
        return HandleGetRequest(request);
      }
      else if (IsValidEditRequest(request)) {
        return HandleEditRequest(request);
      }
      else if (IsValidPingRequest(request)) {
        return HandlePingRequest(request);
      }
      else {
        Syslog(LOG_ERR, "Invalid request");
        return nlohmann::json();
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Responder::Run()
    {
#if 0
      set_pthread_name(_thread.native_handle(), "responder");
#endif
      Syslog(LOG_DEBUG, "Responder started for %s",
             _peer.EndPointString().c_str());
      if (_peer.Authenticate(_server.GetKeyStash(), _server.GetKnownKeys())) {
        RequestMessage  reqmsg;
        while (_run) {
          if (reqmsg.Read(_peer)) {
            ResponseMessage  respmsg(HandleRequest(reqmsg.Json()));
            if (! respmsg.Write(_peer)) {
              Syslog(LOG_ERR, "Failed to send response message to %s at %s",
                     _peer.Id().c_str(), _peer.EndPointString().c_str());
              break;
            }
          }
          else {
            Syslog(LOG_ERR, "Failed to receive message from %s at %s",
                   _peer.Id().c_str(), _peer.EndPointString().c_str());
            break;
          }
        }
        Syslog(LOG_INFO, "Done with client %s at %s",
               _peer.Id().c_str(), _peer.EndPointString().c_str());
      }
      _peer.Disconnect();
      Syslog(LOG_DEBUG, "Responder done for %s",
             _peer.EndPointString().c_str());
      _running = false;
      return;
    }
    
  }  // namespace McBlock

}  // namespace Dwm
