//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcBlockServerConfig.cc 9275 2017-04-26 04:54:13Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockServerConfig.cc
//!  \brief Dwm::McBlock::ServerConfig class implementation
//---------------------------------------------------------------------------

#include "DwmSvnTag.hh"
#include "DwmMcBlockServerConfig.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcblock/tags/mcblock-0.1.1/classes/src/DwmMcBlockServerConfig.cc 9275 $");

using namespace std;

namespace Dwm {

  namespace McBlock {
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ServerConfig::ServerConfig(const Cfg2Json & config)
        : _pfdevpath("/dev/pf"), _tcpAddr(INADDR_ANY), _tcpPort(1001),
          _mcastAddr("224.0.0.42"), _mcastPort(1001),
          _privateKeyFile("/usr/local/etc/mcblockd/id_rsa"),
          _authorizedKeysFile("/usr/local/etc/mcblockd/authorized_keys"),
          _dwmRDAPPrivateKeyFile("/usr/local/etc/mcblockd/id_rsa"),
          _knownServicesFile("/usr/local/etc/mcblockd/known_services"),
          _dwmRDAPServer("127.0.0.1"), _databaseDirectory("/etc")
    {
      const Json::Value *jvp = config.Find("Server");
      if (jvp) {
        const Json::Value *p = config.Find(jvp, "PF/Device");
        if (p && p->isString()) {
          _pfdevpath = p->asString();
        }
        p = config.Find(jvp, "TCP/Address");
        if (p && p->isString()) {
          _tcpAddr = p->asString();
        }
        p = config.Find(jvp, "TCP/Port");
        if (p && p->isString()) {
          _tcpPort = stoul(p->asString());
        }
        p = config.Find("Multicast/Address");
        if (p && p->isString()) {
          _mcastAddr = p->asString();
        }
        p = config.Find(jvp, "Multicast/Port");
        if (p && p->isString()) {
          _mcastPort = stoul(p->asString());
        }
        p = config.Find(jvp, "Auth/PrivKeyFile");
        if (p && p->isString()) {
          _privateKeyFile = p->asString();
        }
        p = config.Find(jvp, "Auth/AuthorizedKeysFile");
        if (p && p->isString()) {
          _authorizedKeysFile = p->asString();
        }
        p = config.Find(jvp, "DwmRDAP/PrivKeyFile");
        if (p && p->isString()) {
          _dwmRDAPPrivateKeyFile = p->asString();
        }
        p = config.Find(jvp, "DwmRDAP/KnownServices");
        if (p && p->isString()) {
          _knownServicesFile = p->asString();
        }
        p = config.Find(jvp, "DwmRDAP/Server");
        if (p && p->isString()) {
          _dwmRDAPServer = p->asString();
        }
        p = config.Find(jvp, "DatabaseDirectory");
        if (p && p->isString()) {
          _databaseDirectory = p->asString();
        }
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & ServerConfig::Device() const
    {
      return _pfdevpath;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & ServerConfig::Device(const std::string & path)
    {
      _pfdevpath = path;
      return _pfdevpath;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & ServerConfig::TCPAddr() const
    {
      return _tcpAddr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & ServerConfig::TCPAddr(const Ipv4Address & addr)
    {
      _tcpAddr = addr;
      return _tcpAddr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t ServerConfig::TCPPort() const
    {
      return _tcpPort;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t ServerConfig::TCPPort(uint16_t tcpPort)
    {
      _tcpPort = tcpPort;
      return _tcpPort;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & ServerConfig::MulticastAddr() const
    {
      return _mcastAddr;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address &
    ServerConfig::MulticastAddr(const Ipv4Address & addr)
    {
      _mcastAddr = addr;
      return _mcastAddr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t ServerConfig::MulticastPort() const
    {
      return _mcastPort;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t ServerConfig::MulticastPort(uint16_t port)
    {
      _mcastPort = port;
      return _mcastPort;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & ServerConfig::PrivateKeyFile() const
    {
      return _privateKeyFile;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & ServerConfig::AuthorizedKeysFile() const
    {
      return _authorizedKeysFile;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & ServerConfig::DwmRDAPPrivateKeyFile() const
    {
      return _dwmRDAPPrivateKeyFile;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & ServerConfig::KnownServicesFile() const
    {
      return _knownServicesFile;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & ServerConfig::DwmRDAPServer() const
    {
      return _dwmRDAPServer;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & ServerConfig::DatabaseDirectory() const
    {
      return _databaseDirectory;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value ServerConfig::Json() const
    {
      Json::Value  value;
      value["pfdevpath"] = _pfdevpath;
      value["tcpAddr"] = (string)_tcpAddr;
      value["tcpPort"] = _tcpPort;
      value["mcastAddr"] = (string)_mcastAddr;
      value["mcastPort"] = _mcastPort;
      value["dwmRDAPPrivateKeyFile"] = _dwmRDAPPrivateKeyFile;
      value["knownServicesFile"] = _knownServicesFile;
      value["dwmRDAPServer"] = _dwmRDAPServer;
      value["databaseDirectory"] = _databaseDirectory;
      return value;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value ServerConfig::Json(const Json::Value & value)
    {
      _pfdevpath = value["pfdevpath"].asString();
      _tcpAddr = Ipv4Address(value["tcpAddr"].asString());
      _tcpPort = value["tcpPort"].asUInt();
      _mcastAddr = Ipv4Address(value["mcastAddr"].asString());
      _mcastPort = value["mcastPort"].asUInt();
      _dwmRDAPPrivateKeyFile = value["dwmRDAPPrivateKeyFile"].asString();
      _knownServicesFile = value["knownServicesFile"].asString();
      _dwmRDAPServer = value["dwmRDAPServer"].asString();
      return Json();
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ServerConfig::operator == (const ServerConfig & config) const
    {
      return ((_pfdevpath == config._pfdevpath)
              && (_tcpAddr == config._tcpAddr)
              && (_tcpPort == config._tcpPort)
              && (_mcastAddr == config._mcastAddr)
              && (_mcastPort == config._mcastPort)
              && (_dwmRDAPPrivateKeyFile == config._dwmRDAPPrivateKeyFile)
              && (_knownServicesFile == config._knownServicesFile)
              && (_dwmRDAPServer == config._dwmRDAPServer)
              && (_databaseDirectory == config._databaseDirectory));
    }
    
  }  // namespace McBlock

}  // namespace Dwm
